# coding: utf-8
from __future__ import unicode_literals

import logging

from static_api.listeners import register, BaseListener

from .. import models, dehydrators

log = logging.getLogger(__name__)


@register('django_intranet_stuff.organization')
class OrganizationListener(BaseListener):
    model_class = models.office.OrganizationModel
    delete_on_zero_intranet_status = False

    def on_modify(self, model, namespace):
        organization = namespace.organization
        person = namespace.person

        organization.put({'id': model.id},
                         data=model.as_dict(dehydrators.office.OrganizationDehydrator))

        person.put_nested('official.organization',
                          {'official.organization.id': model.id},
                          data=model.as_dict(dehydrators.office.OrganizationRefDehydrator))

    def on_delete(self, model, namespace):
        organization = namespace.organization
        person = namespace.person

        organization.delete({'id': model.id})

        person.delete_nested('official.organization', {'official.organization.id': model.id})


@register('django_intranet_stuff.country')
class CountryListener(BaseListener):
    model_class = models.office.CountryModel
    delete_on_zero_intranet_status = False

    def on_modify(self, model, ns):
        ns.country.put({'id': model.id},
                       data=model.as_dict(dehydrators.office.CountryDehydrator))

        def put_ref(collection, path):
            collection.put_nested(path, {path + '.id': model.id},
                                  data=model.as_dict(dehydrators.office.CountryRefDehydrator))

        put_ref(ns.person, 'location.office.city.country')
        put_ref(ns.person, 'location.table.floor.office.city.country')

        put_ref(ns.city, 'country')
        put_ref(ns.office, 'city.country')
        put_ref(ns.floor, 'office.city.country')
        put_ref(ns.table, 'floor.office.city.country')
        put_ref(ns.room, 'floor.office.city.country')
        put_ref(ns.equipment, 'floor.office.city.country')

    def on_delete(self, model, ns):
        ns.country.delete({'id': model.id})

        def delete_ref(collection, path):
            collection.delete_nested(path, {path + '.id': model.id})

        delete_ref(ns.person, 'location.office.city.country')
        delete_ref(ns.person, 'location.table.floor.office.city.country')

        delete_ref(ns.city, 'country')
        delete_ref(ns.office, 'city.country')
        delete_ref(ns.floor, 'office.city.country')
        delete_ref(ns.table, 'floor.office.city.country')
        delete_ref(ns.room, 'floor.office.city.country')
        delete_ref(ns.equipment, 'floor.office.city.country')


@register('django_intranet_stuff.city')
class CityListener(BaseListener):
    model_class = models.office.CityModel
    delete_on_zero_intranet_status = False

    def on_modify(self, model, ns):
        ns.city.put({'id': model.id},
                    data=model.as_dict(dehydrators.office.CityDehydrator))

        def put_ref(collection, path):
            collection.put_nested(path, {path + '.id': model.id},
                                  data=model.as_dict(dehydrators.office.CityRefDehydrator))

        put_ref(ns.person, 'location.office.city')
        put_ref(ns.person, 'location.table.floor.office.city')

        put_ref(ns.office, 'city')
        put_ref(ns.floor, 'office.city')
        put_ref(ns.table, 'floor.office.city')
        put_ref(ns.room, 'floor.office.city')
        put_ref(ns.equipment, 'floor.office.city')

    def on_delete(self, model, ns):
        ns.city.delete({'id': model.id})

        def delete_ref(collection, path):
            collection.delete_nested(path, {path + '.id': model.id})

        delete_ref(ns.person, 'location.office.city')
        delete_ref(ns.person, 'location.table.floor.office.city')

        delete_ref(ns.office, 'city')
        delete_ref(ns.floor, 'office.city')
        delete_ref(ns.table, 'floor.office.city')
        delete_ref(ns.room, 'floor.office.city')
        delete_ref(ns.equipment, 'floor.office.city')


@register('django_intranet_stuff.office')
class OfficeListener(BaseListener):
    model_class = models.office.OfficeModel
    delete_on_zero_intranet_status = False

    def on_modify(self, model, ns):
        ns.office.put({'id': model.id},
                      data=model.as_dict(dehydrators.office.OfficeDehydrator))

        def put_ref(collection, path):
            collection.put_nested(path, {path + '.id': model.id},
                                  data=model.as_dict(dehydrators.office.OfficeRefDehydrator))

        put_ref(ns.person, 'location.office')
        put_ref(ns.floor, 'office')
        put_ref(ns.table, 'floor.office')
        put_ref(ns.table, 'floor.office')
        put_ref(ns.room, 'floor.office')
        put_ref(ns.equipment, 'office')

    def on_delete(self, model, ns):
        ns.office.delete({'id': model.id})

        def delete_ref(collection, path):
            collection.delete_nested(path, {path + '.id': model.id})

        delete_ref(ns.person, 'location.office')
        delete_ref(ns.floor, 'office')
        delete_ref(ns.table, 'floor.office')
        delete_ref(ns.room, 'floor.office')
        delete_ref(ns.equipment, 'floor.office')


@register('django_intranet_stuff.floor')
class FloorListener(BaseListener):
    model_class = models.office.FloorModel
    delete_on_zero_intranet_status = False

    def on_modify(self, model, ns):
        ns.floor.put({'id': model.id},
                     data=model.as_dict(dehydrators.office.FloorDehydrator))

        def put_ref(collection, path):
            collection.put_nested(path, {path + '.id': model.id},
                                  data=model.as_dict(dehydrators.office.FloorRefDehydrator))

        put_ref(ns.person, 'location.table.floor')

        ns.office.put_nested('floors.$',
                             {'floors.$.id': model.id},
                             parent_lookup={'id': model.office['id']},
                             data=model.as_dict(dehydrators.office.FloorRefNestedDehydrator))

        put_ref(ns.table, 'floor')
        put_ref(ns.room, 'floor')
        put_ref(ns.equipment, 'floor')

    def on_delete(self, model, ns):
        ns.floor.delete({'id': model.id})

        def delete_ref(collection, path):
            collection.delete_nested(path, {path + '.id': model.id})

        delete_ref(ns.person, 'location.table.floor')
        delete_ref(ns.office, 'floors.$')
        delete_ref(ns.table, 'floor')
        delete_ref(ns.room, 'floor')
        delete_ref(ns.equipment, 'floor')


@register('django_intranet_stuff.table')
class TableListener(BaseListener):
    model_class = models.office.TableModel

    def on_modify(self, model, ns):
        ns.table.put({'id': model.id},
                     data=model.as_dict(dehydrators.office.TableDehydrator))

        ns.person.put_nested('location.table',
                             {'location.table.id': model.id},
                             data=model.as_dict(dehydrators.office.TableRefDehydrator))

    def on_delete(self, model, ns):
        ns.table.delete({'id': model.id})

        ns.person.delete_nested('location.table', {'location.table.id': model.id})


@register('django_intranet_stuff.room')
class RoomListener(BaseListener):
    model_class = models.office.RoomModel
    delete_on_zero_intranet_status = False

    def on_modify(self, model, ns):
        ns.room.put({'id': model.id},
                    data=model.as_dict(dehydrators.office.RoomDehydrator))

        def put_ref(collection, path):
            collection.put_nested(path, {path + '.id': model.id},
                                  data=model.as_dict(dehydrators.office.RoomRefDehydrator))

        put_ref(ns.person, 'location.table.room')
        put_ref(ns.person, 'location.room')
        put_ref(ns.table, 'room')

    def on_delete(self, model, ns):
        ns.room.delete({'id': model.id})

        def delete_ref(collection, path):
            collection.delete_nested(path, {path + '.id': model.id})

        delete_ref(ns.person, 'location.table.room')
        delete_ref(ns.person, 'location.room')
        delete_ref(ns.table, 'room')



@register('django_intranet_stuff.device')
class EquipmentListener(BaseListener):
    model_class = models.office.EquipmentModel
    delete_on_zero_intranet_status = False

    def on_modify(self, model, ns):
        if model.get_type() in ('cctv', 'lock'):
            self.on_delete(model, ns)
            return

        ns.equipment.put({'id': model.id},
                         data=model.as_dict(dehydrators.office.EquipmentDehydrator))

    def on_delete(self, model, ns):
        ns.equipment.delete({'id': model.id})
