# coding: utf-8
from __future__ import unicode_literals

import logging

from static_api.listeners import register, BaseListener

from .. import models, dehydrators


log = logging.getLogger(__name__)


@register('django_intranet_stuff.staff')
class PersonListener(BaseListener):
    model_class = models.person.PersonModel
    delete_on_zero_intranet_status = False

    def has_staff_base_roles(self, ns, model):
        base_roles = set(models.group.DepartmentStaffModel.role_mapping.values())
        roles = ns.departmentstaff.get({'person.id': model.id})

        for doc in roles:
            if doc['role'] in base_roles:
                return True

        return False

    def on_modify(self, model, ns):
        ns.person.put({'id': model.id}, data=model.as_dict(dehydrators.person.PersonDehydrator))

        person_ref_data = model.as_dict(dehydrators.person.PersonRefDehydrator)
        person_new_ref_data = model.as_dict(dehydrators.person.PersonNewRefDehydrator)

        def put_ref(collection, path):
            collection.put_nested(path, lookup={path + '.id': model.id}, data=person_ref_data)

        def put_new_ref(collection, path):
            collection.put_nested(path, lookup={path + '.id': model.id}, data=person_new_ref_data)

        def update_nested_ref(collection, path):
            return collection.update_in_nested_array_element(
                path,
                model.id,
                data=person_ref_data
            )

        has_base_roles = self.has_staff_base_roles(ns, model)

        if has_base_roles:
            op = update_nested_ref(ns.department, 'heads.$.person')
            ns.department.execute_bulk([op])

        group_ops = [
            update_nested_ref(ns.group, 'responsibles.$.person'),
            update_nested_ref(ns.group, 'parent.responsibles.$.person'),
            update_nested_ref(ns.group, 'ancestors.$.responsibles.$.person'),
        ]

        if has_base_roles:
            group_ops += [
                update_nested_ref(ns.group, 'department.heads.$.person'),
                update_nested_ref(ns.group, 'parent.department.heads.$.person'),
                update_nested_ref(ns.group, 'ancestors.$.department.heads.$.person'),
            ]

        ns.group.execute_bulk(group_ops, ordered=False)

        person_ops = [
            update_nested_ref(ns.person, 'groups.$.group.responsibles.$.person'),
            update_nested_ref(ns.person, 'groups.$.group.parent.responsibles.$.person'),
            update_nested_ref(ns.person, 'groups.$.group.ancestors.$.responsibles.$.person'),
        ]

        if has_base_roles:
            person_ops += [
                update_nested_ref(ns.person, 'department_group.department.heads.$.person'),
                update_nested_ref(ns.person, 'department_group.parent.department.heads.$.person'),
                update_nested_ref(ns.person, 'department_group.ancestors.$.department.heads.$.person'),
                update_nested_ref(ns.person, 'groups.$.group.department.heads.$.person'),
                update_nested_ref(ns.person, 'groups.$.group.parent.department.heads.$.person'),
                update_nested_ref(ns.person, 'groups.$.group.ancestors.$.department.heads.$.person'),
            ]

        ns.person.execute_bulk(person_ops, ordered=False)

        put_ref(ns.departmentstaff, 'person')
        put_ref(ns.groupmembership, 'person')
        put_ref(ns.groupresponsible, 'person')

        put_new_ref(ns.departmentchain, 'chiefs.$')
        put_new_ref(ns.departmentchain, 'hr_partners.$')
        put_new_ref(ns.person, 'chief')
        put_new_ref(ns.person, 'chiefs.$')
        put_new_ref(ns.person, 'hr_partners.$')

    def on_delete(self, model, ns):
        ns.person.delete({'id': model.id})

        def delete_ref(collection, path):
            collection.delete_nested(path, lookup={path + '.id': model.id})

        delete_ref(ns.department, 'heads.$.person')

        delete_ref(ns.group, 'responsibles.$.person')
        delete_ref(ns.group, 'parent.responsibles.$.person')
        delete_ref(ns.group, 'department.heads.$.person')
        delete_ref(ns.group, 'parent.department.heads.$.person')
        delete_ref(ns.group, 'ancestors.$.responsibles.$.person')
        delete_ref(ns.group, 'ancestors.$.department.heads.$.person')

        delete_ref(ns.person, 'groups.$.group.responsibles.$.person')
        delete_ref(ns.person, 'groups.$.group.parent.responsibles.$.person')
        delete_ref(ns.person, 'groups.$.group.ancestors.$.responsibles.$.person')

        if self.has_staff_base_roles(ns, model):
            delete_ref(ns.person, 'department_group.department.heads.$.person')
            delete_ref(ns.person, 'department_group.parent.department.heads.$.person')
            delete_ref(ns.person, 'department_group.ancestors.$.department.heads.$.person')
            delete_ref(ns.person, 'groups.$.group.department.heads.$.person')
            delete_ref(ns.person, 'groups.$.group.parent.department.heads.$.person')
            delete_ref(ns.person, 'groups.$.group.ancestors.$.department.heads.$.person')

        delete_ref(ns.departmentstaff, 'person')

        delete_ref(ns.groupmembership, 'person')

        delete_ref(ns.groupresponsible, 'person')


@register('django_intranet_stuff.staffphone')
class StaffPhoneListener(BaseListener):
    model_class = models.person.PersonPhoneModel

    def on_modify(self, model, ns):
        if model.kind in ('emergency', 'hidden'):
            self.on_delete(model, ns)
            return

        ns.personphone.put({'id': model.id},
                           data=model.as_dict(dehydrators.person.PersonPhoneDehydrator))

        ns.person.put_nested('phones.$',
                             lookup={'phones.$.id': model.id},
                             parent_lookup={'id': model.person},
                             data=model.as_dict(dehydrators.person.PersonPhoneRefDehydrator))

    def on_delete(self, model, ns):
        ns.personphone.delete({'id': model.id})

        ns.person.delete_nested('phones.$',
                                lookup={'phones.$.id': model.id})


@register('django_intranet_stuff.staffcar')
class StaffCarListener(BaseListener):
    model_class = models.person.PersonCarModel

    def on_modify(self, model, ns):
        ns.personcar.put({'id': model.id},
                         data=model.as_dict(dehydrators.person.PersonCarDehydrator))

        ns.person.put_nested('cars.$',
                             lookup={'cars.$.id': model.id},
                             parent_lookup={'id': model.person},
                             data=model.as_dict(dehydrators.person.PersonCarRefDehydrator))

    def on_delete(self, model, ns):
        ns.personcar.delete({'id': model.id})

        ns.person.delete_nested('cars.$',
                                lookup={'cars.$.id': model.id})



@register('person.bicycle')
class StaffBicycleListener(BaseListener):
    model_class = models.person.PersonBicycleModel

    def on_modify(self, model, ns):
        ns.personbicycle.put({'id': model.id},
                         data=model.as_dict(dehydrators.person.PersonBicycleDehydrator))

        ns.person.put_nested('bicycles.$',
                             lookup={'bicycles.$.id': model.id},
                             parent_lookup={'id': model.person},
                             data=model.as_dict(dehydrators.person.PersonBicycleRefDehydrator))

    def on_delete(self, model, ns):
        ns.personbicycle.delete({'id': model.id})

        ns.person.delete_nested('bicycles.$',
                                lookup={'bicycles.$.id': model.id})


@register('person.contact')
class StaffAccountListener(BaseListener):
    model_class = models.person.PersonAccountModel

    def on_modify(self, model, ns):
        if not model.is_supported_contact_type():
            log.info('Got unsupported contact type %s', model.get_raw('contact_type'))
            return

        ns.personaccount.put({'id': model.id}, data=model.as_dict(dehydrators.person.PersonAccountDehydrator))

        ns.person.put_nested(
            attr='accounts.$',
            lookup={'accounts.$.id': model.id},
            parent_lookup={'id': model.person},
            data=model.as_dict(dehydrators.person.PersonAccountRefDehydrator),
        )

        account_without_type = model.as_dict(dehydrators.person.PersonAccountRefWithoutTypeDehydrator)

        type_ = model.get_type()
        if type_ == 'personal_email':
            ns.person.put_nested(
                attr='personal_emails.$',
                lookup={'personal_emails.$.id': model.id},
                parent_lookup={'id': model.person},
                data=account_without_type,
            )
        elif type_ == 'telegram':
            ns.person.put_nested(
                attr='telegram_accounts.$',
                lookup={'telegram_accounts.$.id': model.id},
                parent_lookup={'id': model.person},
                data=account_without_type,
            )
        elif type_ == 'skype':
            ns.person.put_nested(
                attr='skype_accounts.$',
                lookup={'skype_accounts.$.id': model.id},
                parent_lookup={'id': model.person},
                data=account_without_type,
            )

    def on_delete(self, model, ns):
        ns.personaccount.delete({'id': model.id})

        ns.person.delete_nested('accounts.$', lookup={'accounts.$.id': model.id})
        ns.person.delete_nested('personal_emails.$', lookup={'personal_emails.$.id': model.id})
        ns.person.delete_nested('telegram_accounts.$', lookup={'telegram_accounts.$.id': model.id})
        ns.person.delete_nested('skype_accounts.$', lookup={'skype_accounts.$.id': model.id})


# должен некоторое время жить, чтобы довычитать старые записи в emission log'е
@register('django_intranet_stuff.staffkey')
class StaffKeyListener(BaseListener):
    model_class = models.person.PersonKeyModel

    def on_modify(self, model, ns):
        ns.personkey.put({'id': model.id},
                         data=model.as_dict(dehydrators.person.PersonKeyDehydrator))

        ns.person.put_nested('keys.$',
                             lookup={'keys.$.id': model.id},
                             parent_lookup={'id': model.person},
                             data=model.as_dict(dehydrators.person.PersonKeyRefDehydrator))

    def on_delete(self, model, ns):
        ns.personkey.delete({'id': model.id})

        ns.person.delete_nested('keys.$',
                                lookup={'keys.$.id': model.id})


@register('keys.sshkey')
class SSHKeyListener(BaseListener):
    model_class = models.person.PersonKeyModel

    def on_modify(self, model, ns):
        ns.personkey.put({'id': model.id},
                         data=model.as_dict(dehydrators.person.PersonKeyDehydrator))

        ns.person.put_nested('keys.$',
                             lookup={'keys.$.id': model.id},
                             parent_lookup={'id': model.person},
                             data=model.as_dict(dehydrators.person.PersonKeyRefDehydrator))

    def on_delete(self, model, ns):
        ns.personkey.delete({'id': model.id})

        ns.person.delete_nested('keys.$',
                                lookup={'keys.$.id': model.id})


@register('keys.gpgkey')
class GpgKeyListener(BaseListener):
    model_class = models.person.PersonGpgKeyModel

    def on_modify(self, model, ns):
        ns.persongpgkey.put({'id': model.id},
                            data=model.as_dict(dehydrators.person.PersonGpgKeyDehydrator))

        ns.person.put_nested('gpg_keys.$',
                             lookup={'gpg_keys.$.id': model.id},
                             parent_lookup={'id': model.person},
                             data=model.as_dict(dehydrators.person.PersonGpgKeyRefDehydrator))

    def on_delete(self, model, ns):
        ns.persongpgkey.delete({'id': model.id})

        ns.person.delete_nested('gpg_keys.$',
                                lookup={'gpg_keys.$.id': model.id})


@register('emails.email')
class EmailListener(BaseListener):
    model_class = models.person.PersonEmailModel

    def on_modify(self, model, ns):
        ns.personemail.put({'id': model.id},
                           data=model.as_dict(dehydrators.person.PersonEmailDehydrator))

        ns.person.put_nested('emails.$',
                             lookup={'emails.$.id': model.id},
                             parent_lookup={'id': model.person},
                             data=model.as_dict(dehydrators.person.PersonEmailRefDehydrator))

    def on_delete(self, model, ns):
        ns.personemail.delete({'id': model.id})

        ns.person.delete_nested('emails.$',
                                lookup={'emails.$.id': model.id})


@register('django_intranet_stuff.memorialprofile')
class MemorialProfileListener(BaseListener):
    model_class = models.person.MemorialProfileModel

    def on_modify(self, model, ns):
        ns.personmemorial.put({'id': model.id},
                              data=model.as_dict(dehydrators.person.MemorialProfileDehydrator))

        ns.person.put_nested('memorial',
                             lookup={'memorial.id': model.id},
                             parent_lookup={'id': model.person},
                             data=model.as_dict(dehydrators.person.MemorialProfileRefDehydrator))

    def on_delete(self, model, ns):
        ns.personmemorial.delete({'id': model.id})

        ns.person.delete_nested('memorial',
                                lookup={'memorial.id': model.id})


@register('person.staffextrafields')
class PersonExtraFieldsListener(BaseListener):
    model_class = models.person.PersonExtraFieldsModel

    def on_modify(self, model, ns):
        ns.personextra.put({'id': model.id},
                           data=model.as_dict(dehydrators.person.PersonExtraFieldsDehydrator))

        ns.person.put({'id': model.id},
                      data=model.as_dict(dehydrators.person.PersonExtraFieldsPersonDehydrator),
                      merge=True)

    def on_delete(self, model, ns):
        ns.personextra.delete({'id': model.id})


@register('person.responsibleforrobot')
class PersonResponsibleForRobotListener(BaseListener):
    model_class = models.person.PersonResponsibleForRobotModel

    def on_modify(self, model, ns):
        ns.responsibleforrobot.put(
            {'id': model.id},
            data=model.as_dict(dehydrators.person.PersonResponsibleForRobotDehydrator)
        )

        if model.role == 'owner':
            ns.person.put_nested(
                'robot_owners.$',
                lookup={'robot_owners.$.id': model.id},
                parent_lookup={'id': model.robot},
                data=model.as_dict(dehydrators.person.PersonResponsibleForRobotRefDehydrator)
            )
            return

        if model.role == 'user':
            ns.person.put_nested(
                'robot_users.$',
                lookup={'robot_users.$.id': model.id},
                parent_lookup={'id': model.robot},
                data=model.as_dict(dehydrators.person.PersonResponsibleForRobotRefDehydrator)
            )
            return

        log.error('Unexpected robot responsible role')

    def on_delete(self, model, ns):
        ns.responsibleforrobot.delete({'id': model.id})

        if model.role == 'owner':
            ns.person.delete_nested(
                'robot_owners.$',
                lookup={'robot_owners.$.id': model.id}
            )
            return

        if model.role == 'user':
            ns.person.delete_nested(
                'robot_users.$',
                lookup={'robot_users.$.id': model.id}
            )
            return

        log.error('Unexpected robot responsible role')
