# coding: utf-8
from __future__ import unicode_literals

import logging

from django.conf import settings

from static_api.models import (
    Model,
    memoize,
    NativeLangMixin,
    IsDeletedMixin,
)

from .person import DepartmentGroupMixin

log = logging.getLogger(__name__)


ROOT_PARENT_IDS = [1, 2, 3, 4]


class DepartmentModel(IsDeletedMixin, NativeLangMixin, Model):
    def get_name(self):
        return {
            'full': {
                'ru': self.get_raw('name'),
                'en': self.get_raw('name_en'),
            },
            'short': {
                'ru': self.get_raw('short_name'),
                'en': self.get_raw('short_name_en'),
            }
        }

    def get_description(self):
        return {
            'ru': self.get_raw('description'),
            'en': self.get_raw('description_en'),
        }

    def get_contacts(self):
        return {
            'maillists': filter(
                None, self.get_raw('maillists', '').split(',')
            ),
            'wiki': self.get_raw('wiki_page'),
        }

    @memoize
    def get_heads(self):
        return self.namespace.departmentstaff.get({'department_group.department.id': self.id})

    @memoize
    def get_kind(self):
        kind_id = self.get_raw('kind')
        return settings.API_DEPARTMET_KINDS.get(kind_id)


class DepartmentKindModel(NativeLangMixin, Model):
    def get_name(self):
        return {
            'ru': self.get_raw('name'),
            'en': self.get_raw('name_en')
        }


class DepartmentChainModel(Model):
    def get_department(self):
        return self.get_raw('department')

    def get_chiefs(self):
        chiefs = self.get_raw('chiefs')
        docs = self.namespace.person.get({'id': {'$in': chiefs}})
        order = {chief: index for index, chief in enumerate(chiefs)}
        return sorted(docs, key=lambda v: order[v['id']])

    def get_hr_partners(self):
        hr_partners = self.get_raw('hr_partners')
        docs = self.namespace.person.get({'id': {'$in': hr_partners}})
        order = {partner: index for index, partner in enumerate(hr_partners)}
        return sorted(docs, key=lambda v: order[v['id']])


class GroupModel(IsDeletedMixin, Model):
    type_mapping = {
        0: 'wiki',
        1: 'department',
        2: 'service',
        3: 'servicerole',
    }

    def get_service(self):
        return {
            'id': (self.get_raw('service')
                   or self.get_raw('service_id')),
        }

    def get_role_scope(self):
        return self.get_raw('role_scope_id')

    def get_type(self):
        type_ = self.get_raw('type')

        try:
            type_ = int(type_)
        except (ValueError, TypeError):
            pass

        return self.type_mapping.get(type_, type_)

    def get_affiliation_counters(self):
        return {
            'external': self.get_raw('externals_count', 0),
            'yandex': self.get_raw('yandex_count', 0),
            'yamoney': self.get_raw('yamoney_count', 0),
        }

    @memoize
    def get_department(self):
        id_ = (self.get_raw('department')
               or self.get_raw('department_id'))

        if id_:
            docs = self.namespace.department.get({'id': id_})

            if docs:
                return docs[0]

            log.warning('Cannot find Department: %s', id_)

            return {'id': id_}
        else:
            return {'id': None}

    @memoize
    def get_parent(self):
        parent_id = self.get_raw('parent') or self.get_raw('parent_id')

        if parent_id in ROOT_PARENT_IDS:
            parent_id = None

        result = None

        if parent_id:
            parent = self.namespace.group.get_one({'id': parent_id})
            if parent:
                result = parent
            else:
                log.error('Cannot find group parent: %s', parent_id)
                raise ValueError('Group refers to non existent parent')

        return result

    @memoize
    def get_ancestors(self):
        ancestors = []
        if self.parent:
            ancestors = self.parent.get('ancestors', [])
            if self.parent['id'] and self.parent['id'] not in ROOT_PARENT_IDS:
                ancestors += [self.parent]
        return ancestors

    @memoize
    def get_responsibles(self):
        return self.namespace.groupresponsible.get({'group': self.id})


class PersonMixin(object):
    @memoize
    def get_person(self):
        id_ = (self.get_raw('staff')
               or self.get_raw('staff_id'))

        if id_:
            docs = self.namespace.person.get({'id': id_})

            if docs:
                return docs[0]

            log.warning('Cannot find Staff: %s', id_)

            return {'id': id_}
        else:
            return {'id': None}


class GroupMixin(object):
    @memoize
    def get_group(self):
        id_ = (self.get_raw('group')
               or self.get_raw('group_id'))

        if id_:
            docs = self.namespace.group.get({'id': id_})

            if docs:
                return docs[0]

            log.warning('Cannot find Group: %s', id_)

            return {'id': id_}
        else:
            return {'id': None}


class GroupMembershipModel(PersonMixin, GroupMixin, Model):
    @memoize
    def get_joined_at(self):
        return self.convert_time(self.get_raw('joined_at'))


class GroupResponsibleModel(PersonMixin, Model):
    @memoize
    def get_permitted_at(self):
        return self.convert_time(self.get_raw('permitted_at'))


class DepartmentStaffModel(PersonMixin, DepartmentGroupMixin, Model):
    role_mapping = {
        'C': 'chief',
        'c': 'chief',
        'D': 'deputy',
        'd': 'deputy',
        'H': 'hr_partner',
        'h': 'hr_partner',
        'B': 'budget_holder',
        'b': 'budget_holder',
        'G': 'general_director',
        'g': 'general_director',
        'A': 'hr_analyst',
        'a': 'hr_analyst',
        'RECRUITMENT_PARTNER': 'recruitment_partner',
        'AT': 'hr_analyst_temp',
    }

    def get_role(self):
        role = self.get_raw('role')

        return self.role_mapping.get(role, role)
