# coding: utf-8
from __future__ import unicode_literals

import logging

from django.conf import settings

from static_api.models import Model, memoize, NativeLangMixin, IsDeletedMixin


log = logging.getLogger(__name__)


class CreatedAtMixin(object):
    def get_created_at(self):
        return self.convert_time(self.get_raw('created_at'))


class OrganizationModel(IsDeletedMixin, CreatedAtMixin, Model):
    pass


class CountryModel(IsDeletedMixin, CreatedAtMixin, NativeLangMixin, Model):
    def get_name(self):
        return {
            'ru': self.get_raw('name'),
            'en': self.get_raw('name_en'),
        }

    def get_geobase(self):
        return {
            'id': self.get_raw('geo_base_id')
        }

    def get_domain(self):
        return self.get_raw('yandex_domain')


class CityModel(IsDeletedMixin, CreatedAtMixin, NativeLangMixin, Model):
    def get_name(self):
        return {
            'ru': self.get_raw('name'),
            'en': self.get_raw('name_en'),
        }

    @memoize
    def get_country(self):
        id_ = (self.get_raw('country')
               or self.get_raw('country_id'))

        if id_:
            docs = self.namespace.country.get({'id': id_})

            if docs:
                return docs[0]

            log.warning('Cannot find Country: %s', id_)

            return {'id': id_}
        else:
            return {'id': None}


class OfficeModel(IsDeletedMixin, CreatedAtMixin, NativeLangMixin, Model):
    def get_name(self):
        return {
            'ru': self.get_raw('name'),
            'en': self.get_raw('name_en'),
        }

    def get_contacts(self):
        return {
            'phone': self.get_raw('phone'),
            'fax': self.get_raw('fax'),
        }

    def get_address(self):
        return {
            'ru': ' '.join([self.get_raw('address1'), self.get_raw('address2')]),
            'en': ' '.join([self.get_raw('address1_en'), self.get_raw('address2_en')]),
        }

    def get_timezone(self):
        return self.get_raw('tz')

    @memoize
    def get_city(self):
        id_ = (self.get_raw('city')
               or self.get_raw('city_id'))

        if id_:
            docs = self.namespace.city.get({'id': id_})

            if docs:
                return docs[0]

            log.warning('Cannot find City: %s', id_)

            return {'id': id_}
        else:
            return {'id': None}

    @memoize
    def get_floors(self):
        return self.namespace.floor.get({'office.id': self.id})


class FloorModel(IsDeletedMixin, CreatedAtMixin, NativeLangMixin, Model):
    def get_name(self):
        return {
            'ru': self.get_raw('name'),
            'en': self.get_raw('name_en'),
        }

    def get_number(self):
        return self.get_raw('num')

    @memoize
    def get_office(self):
        id_ = (self.get_raw('office')
               or self.get_raw('office_id'))

        if id_:
            docs = self.namespace.office.get({'id': id_})

            if docs:
                return docs[0]

            log.warning('Cannot find Office: %s', id_)

            return {'id': id_}
        else:
            return {'id': None}

    def get_image(self):
        img = self.get_raw('img')

        if not img:
            return None

        return settings.FLOOR_IMAGES_URL + img


class FloorMixin(object):
    @memoize
    def get_floor(self):
        id_ = (self.get_raw('floor')
               or self.get_raw('floor_id'))

        if id_:
            docs = self.namespace.floor.get({'id': id_})

            if docs:
                return docs[0]

            log.warning('Cannot find Floor: %s', id_)

            return {'id': id_}
        else:
            return {'id': None}


class TableModel(FloorMixin, CreatedAtMixin, Model):
    def get_number(self):
        # Пока возвращаем это поле для обратной совместимости
        return self.get_id()

    def get_location(self):
        return {
            'x': self.get_raw('coord_x'),
            'y': self.get_raw('coord_y'),
        }

    @memoize
    def get_room(self):
        id_ = self.get_raw('room') or self.get_raw('room_id')

        if id_:
            docs = self.namespace.room.get({'id': id_})

            if docs:
                return docs[0]

            log.warning('Cannot find Room: %s', id_)

            return {'id': id_}
        else:
            return {'id': None}


class RoomModel(IsDeletedMixin, FloorMixin, CreatedAtMixin, Model):
    type_mapping = {
        0: 'office',
        1: 'conference',
        2: 'smoking',
        3: 'coffee',
        4: 'wc',
        5: 'wardrobe',
        6: 'warehouse',
        7: 'service',
        8: 'shower',
        9: 'gym',
        10: 'library',
        11: 'coworking',
        12: 'stairs',
        13: 'switch',
        14: 'hallway',
        15: 'balcony',
        16: 'phone_booth',
    }

    def get_type(self):
        type_ = self.get_raw('room_type')

        try:
            type_ = int(type_)
        except (ValueError, TypeError):
            pass

        return self.type_mapping.get(type_, type_)

    def get_name(self):
        return {
            'display': self.get_raw('name'),
            'ru': self.get_raw('name'),
            'en': self.get_raw('name_en'),
            'exchange': self.get_raw('name_exchange'),
            'alternative': self.get_raw('name_alternative'),
        }

    def get_is_bookable(self):
        return self.get_raw('is_avallable_for_reserve')

    def get_location(self):
        return {
            'x': self.get_raw('coord_x'),
            'y': self.get_raw('coord_y')
        }

    def get_number(self):
        return self.get_raw('num')

    def get_equipment(self):
        return {
            'screen': self.get_raw('panel') or 0,
            'seats': self.get_raw('seats') or 0,
            'projector': self.get_raw('projector') or 0,
            'desk': self.get_raw('desk'),
            'game_console': self.get_raw('game_console') or '',
            'marker_board': self.get_raw('marker_board'),
            'cork_board': self.get_raw('cork_board'),
            'video_conferencing': self.get_raw('video_conferencing') or '',
            'voice_conferencing': self.get_raw('voice_conferencing'),
            'guest_wifi': self.get_raw('guest_wifi'),
        }


class EquipmentModel(IsDeletedMixin, FloorMixin, CreatedAtMixin, Model):
    type_mapping = {
        0: 'printer',
        1: 'scanner',
        2: 'copier',
        3: 'mfp',
        4: 'dect',
        5: 'wifi',
        6: 'rj45',
        7: 'fax',
        8: 'shredder',
        9: 'cctv',
        10: 'stationery_cabinet',
        11: 'lock',
        12: 'hypercube',
        13: 'sensor',
        14: 'zombie',
        15: 'wifi_mount_platform',
        16: 'wifi_sensor',
        17: 'vending_machine',
        18: 'help_machine',
        19: 'atm',
        20: 'massage_chair',
        21: 'battery_recycling_container',
        22: 'sleep_capsule',
        23: 'photo_booth',
        24: 'photo_spot'
    }

    def get_location(self):
        return {
            'x': self.get_raw('coord_x'),
            'y': self.get_raw('coord_y'),
        }

    def get_type(self):
        type_ = self.get_raw('type')

        try:
            type_ = int(type_)
        except (ValueError, TypeError):
            pass

        return self.type_mapping.get(type_, type_)

    def get_hostname(self):
        return self.get_raw('name_dns')
