# coding: utf-8
from __future__ import unicode_literals

import logging

from django.conf import settings

from static_api.models import (
    Model,
    memoize,
    NativeLangMixin,
    IsDeletedMixin,
)


log = logging.getLogger(__name__)


class DepartmentGroupMixin(object):
    @memoize
    def get_department_group(self):
        id_ = (self.get_raw('department')
               or self.get_raw('department_id'))

        if id_:
            docs = self.namespace.group.get({'department.id': id_})

            if docs:
                return docs[0]

            log.warning('Cannot find Department: %s', id_)

            return {'department': {'id': id_}}
        else:
            return {'department': {'id': None}}


class PersonModel(IsDeletedMixin, DepartmentGroupMixin, NativeLangMixin, Model):
    links_mapping = (
        ('home_page',     'site'),
        ('icq',           'icq'),
        ('login_mk',      'mk'),
        ('login_twitter', 'twitter'),
        ('login_lj',      'blog'),
        ('jabber',        'jabber'),
        ('home_email',    'email'),
        ('login_skype',   'skype'),
    )

    edu_status_mapping = {
        'S': 'secondary',
        'I': 'incomplete',
        'B': 'bachelor',
        'M': 'master',
        'P': 'specialist',
        'A': 'academic',
    }

    edu_direction_mapping = {
        'T': 'technical',
        'L': 'liberal',
        'N': 'natural',
        'E': 'economic',
        'B': 'business',
    }

    gender_mapping = {
        'M': 'male',
        'F': 'female',
    }

    family_status_mapping = {
        'S': 'single',
        'I': 'single',
        'M': 'married',
        'A': 'married',
    }

    employment_mapping = {
        'F': 'full',
        'P': 'partial',
        'D': 'secondary',
    }

    affiliation_mapping = {}

    def get_created_at(self):
        return self.convert_time(self.get_raw('created_at'))

    def get_contacts(self):
        links = []

        for (field, type_) in self.links_mapping:
            value = self.get_raw(field)
            if not value:
                continue

            links.append({'value': self.get_raw(field), 'type': type_})

        return links

    def get_name(self):
        hide_middle_name = self.data['fields']['hide_middle_name']

        return {
            'first': {
                'ru': self.data['fields']['first_name'],
                'en': self.data['fields']['first_name_en'],
            },
            'last': {
                'ru': self.data['fields']['last_name'],
                'en': self.data['fields']['last_name_en'],
            },
            'middle': self.data['fields']['middle_name'] if not hide_middle_name else '',
            'has_namesake': self.data['fields']['has_namesake'],
            'hidden_middle': hide_middle_name,
        }

    def get_location(self):
        return {
            'room': self._get_room(),
            'table': self._get_table(),
            'office': self._get_office(),
            'description': {
                'ru': self.data['fields']['location_descr'],
                'en': self.data['fields']['location_descr_en'],
            }
        }

    def get_personal(self):
        gender = self.data['fields']['gender']
        family_status = self.data['fields']['family_status']

        return {
            'address': {
                'ru': self.data['fields']['address'],
                'en': self.data['fields']['address_en'],
            },
            'birthday': self.data['fields']['birthday'],
            'gender': self.gender_mapping.get(gender, gender),
            'family_status': self.family_status_mapping.get(family_status, family_status),
            'children': self.data['fields']['children'] or 0,
            'about': self.data['fields']['about'],
            'tshirt_size': self.data['fields']['tshirt_size'],
            'mobile_phone_model': self.data['fields']['mobile_phone_model'],
        }

    def get_education(self):
        status = self.data['fields']['edu_status']
        direction = self.data['fields']['edu_direction']

        return {
            'status': self.edu_status_mapping.get(status, status),
            'area': self.edu_direction_mapping.get(direction, direction),
            'place': {
                'ru': self.data['fields']['edu_place'],
                'en': self.data['fields']['edu_place_en'],
            },
            'date': self.data['fields']['edu_date'],
        }

    def get_official(self):
        employment = self.data['fields']['employment']
        affiliation = self.data['fields']['affiliation']
        extra = self._get_extra()

        return dict({
            'duties': {
                'ru': self.data['fields']['duties'],
                'en': self.data['fields']['duties_en'],
            },
            'organization': self._get_organization(),
            'employment': self.employment_mapping.get(employment, employment),
            'position': {
                'ru': self.data['fields']['position'],
                'en': self.data['fields']['position_en'],
            },
            'join_at': self.data['fields']['join_at'],
            'quit_at': self.data['fields']['quit_at'],
            'is_dismissed': self.data['fields']['is_dismissed'],
            'is_homeworker': self.data['fields']['is_homeworker'],
            'is_robot': self.data['fields']['is_robot'],
            'affiliation': self.affiliation_mapping.get(affiliation, affiliation),
            'is_trainee': self.data['fields']['date_completion_internship'] is not None,
        }, **extra)

    def get_language(self):
        return {
            'ui': self.data['fields']['lang_ui'],
            'content': self.data['fields']['lang_content'],
            'native': self.data['fields']['native_lang'],
            'auto_translate': self.data['fields']['auto_translate'],
        }

    def get_environment(self):
        return {
            # 'computer': self.data['fields']['computer'],
            'shell': self.data['fields']['shell'],
            'timezone': self.data['fields']['tz'],
        }

    def get_images(self):
        login = self.get_raw('login')
        host = settings.CENTER_API_HOST

        return {
            'photo': u'https://{host}/api/v1/user/{login}/photo.jpg'.format(login=login, host=host),
            'avatar': u'https://{host}/api/v1/user/{login}/avatar.jpg'.format(login=login, host=host),
        }

    def get_yandex(self):
        return {
            'login': self.get_raw('login_passport'),
        }

    @memoize
    def _get_departmentchain(self):
        department_id = self.get_raw('department')
        return self.namespace.departmentchain.get_one({'department': department_id}) or {}

    @memoize
    def get_chiefs(self):
        chiefs = self._get_departmentchain().get('chiefs', [])
        if chiefs and chiefs[0]['login'] == self.get_raw('login'):
            chiefs = chiefs[1:]

        return chiefs

    def get_chief(self):
        chiefs = self.get_chiefs()
        return chiefs[0] if chiefs else None

    def get_hr_partners(self):
        hr_partners = self._get_departmentchain().get('hr_partners', [])
        return hr_partners

    @memoize
    def get_phones(self):
        return self.namespace.personphone.get({'person': self.id})

    @memoize
    def get_cars(self):
        return self.namespace.personcar.get({'person': self.id})

    @memoize
    def get_bicycles(self):
        return self.namespace.personbicycle.get({'person': self.id})

    @memoize
    def get_accounts(self):
        return self.namespace.personaccount.get({'person': self.id})

    @memoize
    def get_keys(self):
        return self.namespace.personkey.get({'person': self.id})

    @memoize
    def get_gpg_keys(self):
        return self.namespace.persongpgkey.get({'person': self.id})

    @memoize
    def get_emails(self):
        return self.namespace.personemail.get({'person': self.id})

    @memoize
    def get_memorial(self):
        return self.namespace.personmemorial.get_one({'person': self.id})

    @memoize
    def get_robot_users(self):
        return self.namespace.responsibleforrobot.get({'robot': self.id, 'role': 'user'})

    @memoize
    def get_robot_owners(self):
        return self.namespace.responsibleforrobot.get({'robot': self.id, 'role': 'owner'})

    @memoize
    def _get_organization(self):
        id_ = (self.get_raw('organization')
               or self.get_raw('organization_id'))

        if id_:
            docs = self.namespace.organization.get({'id': id_})

            if docs:
                return docs[0]

            log.warning('Cannot find Organization: %s', id_)

            return {'id': id_}
        else:
            return {'id': None}

    @memoize
    def _get_table(self):
        id_ = (self.get_raw('table')
               or self.get_raw('table_id'))

        if id_:
            docs = self.namespace.table.get({'id': id_})

            if docs:
                return docs[0]

            log.warning('Cannot find Table: %s', id_)

            return {'id': id_}
        else:
            return {'id': None}

    @memoize
    def _get_room(self):
        id_ = self.get_raw('room') or self.get_raw('room_id')

        if id_:
            docs = self.namespace.room.get({'id': id_})

            if docs:
                return docs[0]

            log.warning('Cannot find Room: %s', id_)

            return {'id': id_}
        else:
            return {'id': None}

    @memoize
    def _get_office(self):
        id_ = (self.get_raw('office')
               or self.get_raw('office_id'))

        if id_:
            docs = self.namespace.office.get({'id': id_})

            if docs:
                return docs[0]

            log.warning('Cannot find Office: %s', id_)

            return {'id': id_}
        else:
            return {'id': None}

    @memoize
    def _get_extra(self):
        # FIXME: очень не хорошее место. Логику получения дополнительных
        #  данных для моделеи надо как-то обобщить и упорядочить
        doc = self.namespace.personextra.get_one({'id': self.id},
                                                  fields={'id': 0, '_id': 0})

        return doc or {
            'contract_ended_at': None,
            'nda_ended_at': None,
            'has_byod_access': False,
            'wiretap': False,
            'staff_agreement': False,
            'staff_biometric_agreement': False,
        }


class PersonPhoneModel(Model):
    type_mapping = {
        2: 'mobile',
        3: 'home',
    }

    def get_person(self):
        return self.get_raw('staff')

    def get_type(self):
        type_ = self.get_raw('type')

        try:
            type_ = int(type_)
        except (ValueError, TypeError):
            pass

        return self.type_mapping.get(type_, type_)

    def get_is_main(self):
        return self.get_raw('position') == 0


class PersonCarModel(Model):
    def get_person(self):
        return self.get_raw('staff')


class PersonBicycleModel(Model):
    def get_person(self):
        return self.get_raw('owner')


class PersonKeyModel(Model):
    def get_person(self):
        return self.get_raw('staff')


class PersonGpgKeyModel(Model):
    def get_person(self):
        return self.get_raw('staff')


class PersonEmailModel(Model):
    def get_address(self):
        return self.get_raw('email')


class MemorialProfileModel(Model):
    pass


class PersonExtraFieldsModel(Model):
    def get_id(self):
        return self.get_raw('staff') or self.get_raw('staff_id')

    def get_nda_ended_at(self):
        return self.get_raw('nda_end_date')

    def get_contract_ended_at(self):
        return self.get_raw('contract_end_date')

    def get_has_byod_access(self):
        return self.get_raw('byod_access')

    def get_wiretap(self):
        return self.get_raw('wiretap')

    def get_staff_agreement(self):
        return self.get_raw('staff_agreement')

    def get_staff_biometric_agreement(self):
        return self.get_raw('staff_biometric_agreement')


class PersonAccountModel(Model):
    type_mapping = {
        1: 'personal_email',
        2: 'jabber',
        3: 'icq',
        4: 'skype',
        5: 'twitter',
        6: 'moi_krug',
        7: 'personal_site',
        8: 'livejournal',
        9: 'another_work_email',
        10: 'github',
        11: 'facebook',
        12: 'vkontakte',
        13: 'habrahabr',
        14: 'instagram',
        15: 'flickr',
        16: 'tumblr',
        17: 'blogspot',
        18: 'telegram',
        19: 'yamb',
        20: 'assistent',
        21: 'apple_id',
        22: 'play_market_id',
        23: 'gmail',
        24: 'slack',
        26: 'steam',
        27: 'meeting_form',
        28: 'nmaps_login',
        29: 'zoom',
        62: 'q',
        95: 'strava',
        128: 'curator_of_external',
    }

    def is_supported_contact_type(self):
        type_ = self.get_raw('contact_type')
        return type_ in self.type_mapping

    def get_person(self):
        return self.get_raw('person')

    def get_type(self):
        type_ = self.get_raw('contact_type')

        return self.type_mapping.get(type_, type_)

    def get_value(self):
        return self.get_raw('account_id')

    def get_value_lower(self):
        return self.get_raw('account_id').lower()


class PersonResponsibleForRobotModel(Model):
    @memoize
    def get_person(self):
        responsible_id = self.get_raw('responsible')

        if responsible_id:
            docs = self.namespace.person.get({'id': responsible_id})

            if docs:
                return docs[0]

            log.warning('Cannot find Staff: %s', responsible_id)

            return {'id': responsible_id}
        else:
            return {'id': None}

    def get_robot(self):
        return self.get_raw('robot')

    def get_role(self):
        return self.get_raw('role')
