# coding: utf-8
from __future__ import unicode_literals

import json
import logging
from collections import defaultdict

from django.conf import settings

log = logging.getLogger(__name__)


class Event(object):
    def __init__(self, id_, action, creation_time, payload):
        self.id = id_
        self.action = action
        self.creation_time = creation_time
        self.payload = payload

    def get_id(self):
        return self.id

    def get_action(self):
        return self.action

    def get_creation_time(self):
        return self.creation_time

    def get_entity(self):
        raise NotImplementedError

    def get_data(self):
        raise NotImplementedError


class Message(object):
    event_class = Event

    def __init__(self, id_, action, creation_time, payload):
        self.id = id_
        self.action = action
        self.creation_time = creation_time
        self.payload = payload

    def __iter__(self):
        for event_payload in self.payload:
            yield self.event_class(self.id, self.action, self.creation_time, event_payload)


class DjangoEvent(Event):
    def get_entity(self):
        return self.payload['model']

    def get_data(self):
        return self.payload


class DjangoMessage(Message):
    event_class = DjangoEvent

    def __init__(self, raw_message):
        id_ = raw_message['id']
        action = raw_message.get('action', 'modify')
        creation_time = raw_message['creation_time']

        data = json.loads(raw_message['data'])

        super(DjangoMessage, self).__init__(id_, action, creation_time, data)


class Listener(object):
    def on_modify(self, event, *args, **kwargs):
        pass

    def on_delete(self, event, *args, **kwargs):
        pass

    def pre_dispatch(self, event, *args, **kwargs):
        pass

    def post_dispatch(self, event, *args, **kwargs):
        pass

    def post_init(self, entities, *args, **kwargs):
        pass

    def dispatch(self, event):
        action = event.get_action()

        if action == 'modify':
            self.on_modify(event)

            return True

        elif action == 'delete':
            self.on_delete(event)

            return True

        return False


class Registry(object):
    def __init__(self):
        self.listeners = defaultdict(list)

    def add_listener(self, entity, listener):
        self.listeners[entity].append(listener)

    def dispatch(self, message, *args, **kwargs):
        log.info('Message (%s) starts processing: %s',
            message.action, message.id)

        handled = False
        has_events = False
        for event in message:
            handled = self.dispatch_event(event, *args, **kwargs) or handled
            has_events = True

        if not has_events:
            log.warning('Message (%s) does not have any events: %s',
                message.action, message.id)

        if handled:
            log.info('Message (%s) handled: %s',
                message.action, message.id)
        else:
            log.warning('Message (%s) not handled: %s',
                message.action, message.id)

        return handled

    def dispatch_init_hooks(self, entities, *args, **kwargs):
        listener_classes = set()
        for entity in entities:
            for cls in self.listeners[entity]:
                listener_classes.add(cls)

        for cls in listener_classes:
            cls().post_init(entities, *args, **kwargs)

    def dispatch_event(self, event, *args, **kwargs):
        listener_classes = self.listeners[event.get_entity()]

        listeners = []
        for listener_class in listener_classes:
            listener = listener_class()
            listeners.append(listener)
            listener.pre_dispatch(event, *args, **kwargs)

        handled = False
        for listener in listeners:
            handled = listener.dispatch(event, *args, **kwargs) or handled

        for listener in listeners:
            listener.post_dispatch(event, *args, **kwargs)

        if not len(listeners):
            log.warning(
                'No listeners found for event: %s:%s',
                event.get_entity(),
                event.get_data()['pk'],
            )

        if handled:
            log.info(
                'Event %s handled: %s:%s',
                event.get_action(),
                event.get_entity(),
                event.get_data()['pk'],
            )
        else:
            log.warning(
                'Event %s not handled: %s:%s',
                event.get_action(),
                event.get_entity(),
                event.get_data()['pk'],
            )

        return handled


