# coding: utf-8
from __future__ import unicode_literals

import re
import logging
import json
import urllib
from email.parser import Parser
from cStringIO import StringIO

import requests

from django.conf import settings
from django.utils import timezone
from django.utils.functional import SimpleLazyObject

from . import utils, tvm2


logger = logging.getLogger(__name__)

STAFF_SERVICE_ALIAS = 'staff'

re_multipart_boundary = re.compile(r'^boundary="([\w\'\(\)\+,\-\./:\=\?]{1,70})"$')


def _create_staff_session():
    staff_session = requests.session()
    staff_session.headers.update({
        tvm2.TVM_SERVICE_TICKET_HEADER: tvm2.get_tvm_ticket_by_deploy(STAFF_SERVICE_ALIAS),
    })
    return staff_session


def multipart_response_iterator(response):
    parser = Parser()

    content_type = response.headers['Content-Type'].split(';')

    if content_type[0].strip().lower() != 'multipart/mixed':
        raise ValueError("Not a multipart message")

    boundary = None
    for ppty in content_type[1:]:
        match = re_multipart_boundary.match(ppty.strip())
        if match:
            boundary = match.group(1)

    if not boundary:
        raise ValueError("Content-Type contains no boundary")

    boundary_starting = "--{}".format(boundary)
    boundary_ending = "--{}--".format(boundary)
    boundaries = (boundary_starting, boundary_ending)

    stream = None
    terminated_correctly = False
    for line in response.iter_lines():
        if line in boundaries:
            if stream:
                stream.seek(0)
                yield parser.parse(stream)
                stream.close()
                stream = None
            if line == boundary_starting:
                stream = StringIO()
            else:
                terminated_correctly = True
        else:
            if stream:
                if len(line) == stream.tell() == 0:
                    # _Иногда_ сразу после разделителя приходит пустая строка,
                    # которая может нам всё испортить.
                    # Кажется, requests некорректно обрабатывает chunk'и ответа
                    continue

                stream.write("{}\n".format(line))

    if stream:
        stream.close()

    if not terminated_correctly:
        raise ValueError("Response is most likely to be incomplete")


def get_unsent_messages(limit):
    url = '{scheme}://{host}/api/emission/get-unsent/'.format(
        scheme=settings.EMISSION_PROTOCOL,
        host=settings.EMISSION_HOST,
        params={'count': limit},
    )
    response = _create_staff_session().get(url)
    response.raise_for_status()

    for message in response.json():
        yield message


def mark_sent(handled_message_ids):
    url = '{scheme}://{host}/api/emission/mark-sent/'.format(
        scheme=settings.EMISSION_PROTOCOL,
        host=settings.EMISSION_HOST,
    )
    response = _create_staff_session().post(url, data={'message_ids': handled_message_ids})
    response.raise_for_status()


def get_events_by_id_range(id_range):
    use_streaming = getattr(settings, 'EMISSION_STREAMING', False)

    options = {}
    url_query = {
        'start': str(id_range[0]),
        'stop': str(id_range[-1]),
    }

    if use_streaming:
        url_query['multipart'] = '1'
        options['stream'] = True

    options['params'] = url_query

    url = '{scheme}://{host}/api/emission/slice/'.format(
        scheme=settings.EMISSION_PROTOCOL,
        host=settings.EMISSION_HOST,
    )

    response = _create_staff_session().get(url, **options)
    response.raise_for_status()

    if use_streaming:
        for part in multipart_response_iterator(response):
            yield json.loads(part.get_payload())
    else:
        for message in response.json():
            yield message


def get_meta(current_id=None):
    url = '{scheme}://{url}/api/emission/meta/'.format(
        scheme=settings.EMISSION_PROTOCOL,
        url=settings.EMISSION_HOST,
    )

    if current_id:
        url += '?current={}'.format(current_id)

    response = _create_staff_session().get(url=url)
    response.raise_for_status()

    return response.json()


def get_entity_data(entity):
    use_streaming = getattr(settings, 'EMISSION_STREAMING', False)
    max_attempts = getattr(settings, 'EMISSION_MAX_ATTEMPTS', 5)

    options = {}
    url_query = {'name': entity}

    if use_streaming:
        url_query['multipart'] = '1'
        options['stream'] = True

    options['params'] = url_query

    url ='{scheme}://{host}/api/emission/entities-data/'.format(
        scheme=settings.EMISSION_PROTOCOL,
        host=settings.EMISSION_HOST,
    )

    for attempt_no in range(1, max_attempts + 1):
        try:
            logger.info('Emission request for `%s`, attempt %s', entity, attempt_no)
            response = _create_staff_session().get(url, **options)
            response.raise_for_status()

            if use_streaming:
                for part in multipart_response_iterator(response):
                    yield json.loads(part.get_payload())
            else:
                yield response.json()
            return
        except requests.exceptions.RequestException as e:
            logger.error('Emission request for `%s` raises: %s', entity, e)

            if e.response and 400 <= e.response.status_code < 500:
                raise

            if attempt_no >= max_attempts:
                raise


def get_current_master_deltas():
    from . import message

    local_last_message = message.manager.get_last_message()

    if not local_last_message:
        return {
            'delta_ids': 0,
            'delta_time': -1.,
        }

    local_last_id = local_last_message['_id']

    meta = get_meta(current_id=local_last_id)
    emission_last_id = meta['last_message']['id']

    delta_ids = emission_last_id - local_last_id
    delta_time = timezone.timedelta(0)

    if delta_ids:
        if 'next_message' in meta:
            next_message = meta['next_message']
            next_timestamp = utils.parse_time(next_message['creation_time'])
            delta_time = timezone.now() - next_timestamp
        else:
            delta_time = timezone.timedelta(seconds=-1.)

    return {
        'delta_ids': delta_ids,
        'delta_time': delta_time.total_seconds(),
    }
