# coding: utf-8
from __future__ import unicode_literals

import logging
from itertools import groupby

from django.conf import settings

from static_api import helpers, storage


log = logging.getLogger(__name__)


class BaseMessageRange(object):
    def get_range(self):
        raise NotImplementedError

    def __iter__(self):
        raise NotImplementedError

    def __eq__(self, other):
        return (type(self) == type(other)
                and self.get_range() == other.get_range())

    def __repr__(self):
        range_ = self.get_range()

        return '%s(%s, %s)' % (self.__class__.__name__, range_[0], range_[-1])


class MessageRange(BaseMessageRange):
    def __init__(self, messages):
        self.messages = messages

    def get_range(self):
        return self.messages[0]['_id'], self.messages[-1]['_id']

    def __iter__(self):
        return iter(self.messages)


class MissingMessageRange(BaseMessageRange):
    def __init__(self, id_range):
        self.id_range = id_range

    def get_range(self):
        return self.id_range[0], self.id_range[-1]

    def __iter__(self):
        return helpers.get_events_by_id_range(self.id_range)


class MessageManager(object):
    @property
    def collection(self):
        return storage.manager.get_meta_namespace()[settings.STATIC_API_MESSAGES_COLLECTION]

    def set_last_message(self, id_, creation_time, trim=False):
        self.collection.put(
            {'_id': id_},
            {
                '_id': id_,
                'time': creation_time,
                'handled': True,
            },
            raw=True
        )
        if trim:
            self.collection.put(
                {'_id': {'$gt': id_}},
                {'handled': False},
            )

    def get_last_message(self):
        return self.collection.get_one(
            {'handled': True},
            sort='-_id',
        )

    def get_unhandled_messages_new(self, limit):
        for message in helpers.get_unsent_messages(limit):
            yield message

        log.info('Yielded all messages from unsent pack')

    def mark_messages_as_sent_new(self, handled_message_ids):
        helpers.mark_sent(handled_message_ids)
        log.info('All handled messages marked as sent')

    def get_unhandled_messages(self, limit):
        last_message = self.get_last_message()

        if last_message:
            last_id = last_message['_id']
        else:
            last_id = 0

        messages = self.collection.get(
            {
                'handled': False,
                '_id': {'$gt': last_id}
            },
            limit=limit, sort='_id'
        )

        for _, g in groupby(enumerate(messages), lambda (i, m): i - m['_id']):
            group = [m for i, m in g]

            first_id = last_id

            if first_id + 1 != group[0]['_id']:
                last_id = min(first_id + limit, group[0]['_id'] - 1)
                limit -= last_id - first_id

                yield MissingMessageRange((first_id + 1, last_id))

                if limit == 0:
                    break

            group = group[:limit]
            limit -= len(group)

            yield MessageRange(group)
            last_id = group[-1]['_id']

            if limit == 0:
                break

        if limit > 0:
            yield MissingMessageRange((last_id + 1, last_id + limit))

    def compact(self, keep=50):
        last_message = self.get_last_message()

        if not last_message:
            return

        self.collection.delete({'_id': {'$lt': last_message['_id'] - keep}})

    def trim_ids(self):
        meta = helpers.get_meta()
        self.collection.delete(
            {'_id': {'$gte': meta['last_message']['id']}}
        )


manager = MessageManager()
