# coding: utf-8
from __future__ import unicode_literals

import functools

from . import utils


def memoize(func):
    @functools.wraps(func)
    def _decorator(self):
        if not hasattr(self, '__memoized__'):
            self.__memoized__ = {}

        key = func.__name__

        try:
            res = self.__memoized__[key]
        except KeyError:
            res = self.__memoized__[key] = func(self)

        return res
    return _decorator


class Model(object):
    empty = object()

    def __init__(self, event, namespace):
        self.event = event
        self.namespace = namespace

        self._dict = None

    @property
    def data(self):
        return self.event.get_data()

    def __getattr__(self, name):
        value = self.get(name, self.empty)

        if value is self.empty:
            raise AttributeError(name)

        return value

    def __iter__(self):
        keys = self.data['fields'].keys()

        keys += [
            attr.split('get_', 1)[1]
            for attr in dir(self.__class__)
            if attr.startswith('get_') and attr not in ('get_raw',)
        ]

        return iter(set(keys))

    def get_raw(self, name, default=None):
        return self.data['fields'].get(name, default)

    def get(self, name, default=None):
        try:
            getter = getattr(self.__class__, 'get_%s' % name)
        except AttributeError:
            value = self.get_raw(name, default)
        else:
            value = getter(self)

        return value

    def get_id(self):
        return self.data['pk']

    def get__meta(self):
        return {
            'message_id': self.event.get_id(),
            'modified_at': self._format_modified_at(self.event.get_creation_time()),
        }

    def as_dict(self, dehydrator=None):
        if self._dict is None:
            self._dict = dict((key, self.get(key)) for key in self)

        dict_ = self._dict

        if dehydrator:
            dict_ = dehydrator(dict_).as_dict()

        return dict_

    def _format_modified_at(self, dt):
        dt_result = utils.parse_time(dt, False)
        return utils.isoformat_with_millis(dt_result)

    def convert_time(self, dt, strip_microseconds=False):
        return utils.parse_time(dt, strip_microseconds).isoformat()


class NativeLangMixin(object):
    def get_native_lang(self):
        return self.get_raw('native_lang') or 'ru'


class IsDeletedMixin(object):
    def get_is_deleted(self):
        return self.get_raw('intranet_status') == 0
