# coding: utf-8
from __future__ import unicode_literals

import re
import jsonschema

from django.core import validators

from static_api import utils


class ParseError(Exception):
    pass


class FilterParamsParser(object):
    KEY_RE = r'^[a-z0-9][a-z0-9_\.]*[a-z0-9]$'
    invalid_value_error_msg = 'Invalid value `%s` for field `%s`'
    unknown_key_error_msg = 'Unknown field `%s`'

    def __init__(self, params, resource):
        self.params = params
        self.resource = resource
        self.cleaned_data = {}
        self.errors = []

    def is_valid(self):
        self.cleaned_data = {}

        for key, value in self.params.items():
            try:
                cleaned_key, cleaned_value = self.validate_pair(key, value)
            except validators.ValidationError as e:
                self.errors.append(str(e.message))
            else:
                self.cleaned_data[cleaned_key] = cleaned_value

        return not self.errors

    def validate_pair(self, key, value):
        if not re.match(self.KEY_RE, key):
            raise validators.ValidationError('Invalid key %s' % key)

        return self.parse_specials(key, value)

    def parse_specials(self, key, val):
        if ',' in val:
            return key, {'$in': [self.parse_specials(key, v)[1]
                                    for v in utils.split_str(val)]}

        field_schema = self.resource.schema.get_field_schema(key)

        if field_schema is None:
            msg = self.unknown_key_error_msg % key
            raise validators.ValidationError(msg)

        field_type = field_schema.get('type')
        enum = field_schema.get('enum')

        if field_type is None and enum is not None:
            try:
                key, val = self.handle_enum(key, val, enum)
            except ParseError:
                msg = self.invalid_value_error_msg % (val, key)
                raise validators.ValidationError(msg)
            return key, val

        if field_type is None:
            field_types = ['string']
        elif isinstance(field_type, basestring):
            field_types = [field_type]
        else:
            field_types = field_type

        # нужно отсортировать допустимые типы, потому что если написать
        # 'string' перед int/null/boolean,
        # то все будет парситься как валидная строка.
        field_types = sorted(field_types, key=lambda type_: [
            'integer',
            'number',
            'null',
            'boolean',
            'string',
            'object',
            'array',
        ].index(type_))

        for type_number, type_ in enumerate(field_types, start=1):
            # по очереди проверяем типы и райзим, только если даже последний
            # парсинг не удался
            try:
                parser = getattr(self, 'parse_' + type_)
                val = parser(val)
            except ParseError:
                if type_number == len(field_types):
                    msg = self.invalid_value_error_msg % (val, key)
                    raise validators.ValidationError(msg)
                else:
                    continue
            else:
                break

        try:
            self.resource.schema.validate_field(key, val)
        except jsonschema.ValidationError as e:
            # what error here?
            raise validators.ValidationError(e)

        return key, val

    def parse_string(self, val):
        if val and val[0] == val[-1] == '"':
            val = val.strip('"')
        return val

    def parse_integer(self, val):
        try:
            return int(val)
        except (ValueError, TypeError):
            raise ParseError()

    def parse_float(self, val):
        try:
            return float(val)
        except (ValueError, TypeError):
            raise ParseError()

    def parse_number(self, val):
        try:
            return self.parse_integer(val)
        except ParseError:
            pass

        # can raise ParseError
        return self.parse_float(val)

    def parse_boolean(self, val):
        if val.lower() == 'true':
            return True
        elif val.lower() == 'false':
            return False

        raise ParseError()

    def parse_null(self, val):
        if not val:
            return None
        if isinstance(val, basestring) and val.lower() in ('null', 'none'):
            return None

        raise ParseError()

    def parse_object(self, val):
        raise ParseError()

    def parse_array(self, val):
        raise ParseError()

    def handle_enum(self, key, val, enum):
        first_enum_element = enum[0]

        if isinstance(first_enum_element, (int, float)):
            val = self.parse_number(val)
        elif isinstance(enum[0], basestring):
            val = self.parse_string(val)

        if val not in enum:
            raise ParseError()
        return key, val
