# coding: utf-8
from __future__ import unicode_literals

import logging
import pql
from collections import OrderedDict

from django import forms
from django.conf import settings
from django.core import validators
from .filter_params import FilterParamsParser


log = logging.getLogger(__name__)


class UnknownField(validators.ValidationError):
    MSG_TEMPLATE = 'Unknown field `%s`'


class UnavailableField(validators.ValidationError):
    MSG_TEMPLATE = 'Unavailable field `%s`'


class ForbiddenField(validators.ValidationError):
    MSG_TEMPLATE = 'Forbidden field `%s`'


class OneOrZeroField(forms.Field):
    """
    Accepts only 0 or 1 in values. Cleans into True or False.
    """
    default_error_messages = {
        'invalid': 'Enter 0 or 1',
    }

    def to_python(self, value):
        if value in validators.EMPTY_VALUES:
            return None
        elif value == '0':
            return False
        elif value == '1':
            return True
        raise validators.ValidationError(self.error_messages['invalid'])


def _query_fields_gen(fields_list):
    if isinstance(fields_list, dict):
        fields_list = [fields_list]
    for fields_dict in fields_list:
        for query_field, value in fields_dict.items():
            if query_field.startswith('$'):
                for nested_field in _query_fields_gen(value):
                    yield nested_field
            else:
                yield query_field


class SpecialParamsParser(forms.Form):

    _doc = OneOrZeroField(required=False)
    _debug = OneOrZeroField(required=False)
    _one = OneOrZeroField(required=False)
    _pretty = OneOrZeroField(required=False)
    _write = OneOrZeroField(required=False)

    _page = forms.IntegerField(min_value=1, required=False)
    _limit = forms.IntegerField(min_value=1, required=False)

    _query = forms.CharField(required=False)

    _sort = forms.CharField(required=False)
    _fields = forms.CharField(required=False)
    _nopage = OneOrZeroField(required=False)
    _explain = OneOrZeroField(required=False)
    _hint = forms.CharField(required=False)

    defaults = {
        '_doc': False,
        '_debug': False,
        '_one': False,
        '_pretty': False,
        '_write': False,
        '_page': 1,
        '_limit': settings.PAGE_SIZE,
        '_nopage': False,
        '_explain': False,
        '_hint': None,
    }

    def __init__(self, resource, permitted_fields, *args, **kwargs):
        self.resource = resource
        self.permitted_fields = permitted_fields
        self.full_access = settings.STATIC_API_WILDCARD_FIELD_ACCESS in self.permitted_fields
        self.has_forbidden_fields = False
        super(SpecialParamsParser, self).__init__(*args, **kwargs)

    def _get_field_schema(self, field):
        return self.resource.schema.get_field_schema(field)

    def _check_field_and_access(self, field):
        field_schema = self._get_field_schema(field)

        if not field_schema:
            raise UnknownField(UnknownField.MSG_TEMPLATE % field)

        if not self._is_permitted_field(field):
            self.has_forbidden_fields = True
            raise ForbiddenField(ForbiddenField.MSG_TEMPLATE % field)

    def _is_permitted_field(self, field):
        return self.full_access or field in self.permitted_fields

    def _filter_excluded(self, fields_scheme, fields):
        for excluded_field, include in fields.items():
            if not include:
                for field in self.resource.schema.hierarchical_index[excluded_field]:
                    fields_scheme.pop(field, None)

        return fields_scheme

    def _get_field_with_all_leaf_fields(self, fields):
        active_fields = self._active_fields(fields)
        if active_fields:
            leaf_fields = {}
            for field in sorted(active_fields):
                leaf_fields.update(self.resource.schema.hierarchical_index[field])
                leaf_fields[field] = self.resource.schema.flat[field]

            return leaf_fields

        leaf_fields = self.resource.schema.flat_leaf
        return self._filter_excluded(leaf_fields, fields)

    def _validate_fields_permissions(self, fields):
        fields = fields or {}
        result = OrderedDict()
        errors = []

        _id_state = fields.pop('_id', None)
        if _id_state is not None:
            result['_id'] = _id_state

        fields = self._get_field_with_all_leaf_fields(fields)
        requested_fields = set(fields)

        # Order is important, in such manner deep fields will be in the beginning
        # e.g department.kind.name before department.kind
        for field in sorted(requested_fields, reverse=True):
            try:
                self._check_field_and_access(field)
            except validators.ValidationError as e:
                errors.append(e)
            else:
                result[field] = 1

        return result, errors

    def clean__query(self):
        errors = []

        query = dict(self.resource.default_filters)

        if self.cleaned_data['_query']:
            try:
                query.update(pql.find(self.cleaned_data['_query']))
                # TODO: restrict complex queries
            except (SyntaxError, pql.ParseError, AttributeError) as e:
                # Иногда даже AttributeError получается из-за баги в pql
                errors.append(validators.ValidationError(e))

        filter_params = {
            key: value
            for key, value in self.data.items()
            if not key.startswith('_')
        }
        filter_params_parser = FilterParamsParser(filter_params, resource=self.resource)
        if not filter_params_parser.is_valid():
            for error in filter_params_parser.errors:
                errors.append(error)

        if errors:
            raise validators.ValidationError(errors)

        query.update(filter_params_parser.cleaned_data)

        for field in self.resource.ignore_fields:
            query.pop(field, None)

        for field in _query_fields_gen(query):
            try:
                self._check_field_and_access(field)
            except validators.ValidationError as e:
                errors.append(e)

        if errors:
            raise validators.ValidationError(errors)

        return query

    def clean__sort(self):
        sort = self.cleaned_data['_sort']

        if not sort:
            return []

        fields = sort.split(',')

        errors = []
        for field in fields:
            stripped_field = field.strip('-+')

            try:
                self._check_field_and_access(stripped_field)
            except validators.ValidationError as e:
                errors.append(e)

        if errors:
            raise validators.ValidationError(errors)

        return fields

    def _active_fields(self, fields):
        return [f for f, is_active in fields.items() if is_active > 0]

    def clean__fields(self):
        fields = self.cleaned_data['_fields']
        one = self.cleaned_data.get('_one', self.defaults['_one'])
        if fields:
            fields = fields.split(',')
        else:
            fields = []

        fields = [f for f in fields if self._get_field_schema(f) or f == '_all']

        fieldset = self.resource.fieldset_cls('one' if one else 'thumbnail')
        # TODO: maybe validate fields instead of ignoring?
        fields = fieldset.process(fields)
        fields, errors = self._validate_fields_permissions(fields)

        if not self.cleaned_data.get('_debug') and errors:
            raise validators.ValidationError(errors)
        elif not fields:
            raise validators.ValidationError('Validation of fields gave an empty list')

        return fields

    def clean__hint(self):
        hint = self.cleaned_data['_hint']

        if hint == 'null':
            return None

        if hint:
            return hint

        if not self.resource.indexes_hint:
            return None

        query = self.cleaned_data.get('_query')
        sort = self.cleaned_data.get('_sort')
        if query is None or sort is None:
            return None

        result = self.resource.indexes_hint(query, sort)
        if result:
            log.debug('Using index %s as hint', result)

        return result

    def clean__limit(self):
        limit = self.cleaned_data['_limit']
        max_limit = settings.STATIC_API_MAX_LIMIT
        if max_limit and limit > max_limit:
            return max_limit
        return limit

    def clean(self):
        cleaned_data = super(SpecialParamsParser, self).clean()

        for field_name, default in self.defaults.items():
            field_value = cleaned_data.get(field_name)
            if not field_value:
                cleaned_data[field_name] = default

        return cleaned_data

    @property
    def cleaned_data_obj(self):
        """
        Чтобы потом везде не тащить за собой префикс-подчеркивание.
        """
        from collections import namedtuple

        cleaned_data = self.cleaned_data
        if not cleaned_data:
            return None

        cls = namedtuple('SpecialParams', [
            'doc',
            'debug',
            'one',
            'pretty',
            'write',
            'page',
            'limit',
            'query',
            'sort',
            'fields',
            'nopage',
            'explain',
            'hint',
        ])

        return cls(**{
            key[len('_'):]: val
            for key, val in cleaned_data.items()
        })
