# coding: utf-8
from __future__ import unicode_literals

from functools import wraps

from . import fields
from .schema import SCHEMAS


class Registry(object):
    data = ({}, {})

    @classmethod
    def register(cls, *resources):
        by_name, by_plural = cls.data
        for resource in resources:
            by_name[resource.name] = resource
            by_plural[resource.plural] = resource

    @property
    def by_name(self):
        return self.data[0]

    @property
    def by_plural(self):
        return self.data[1]


registry = Registry()


class Resource(object):
    required_attrs = (
        'name',
    )
    # in order of dependencies
    optional_attrs = (
        'plural',
        'verbose',
        'list_order',
        'fieldset_cls',
        'default_filters',
        'indexes_hint',
        'ignore_fields',
    )
    attrs = required_attrs + optional_attrs

    def __init__(self, **kwargs):
        # check required params
        required_not_in_kwargs = set(self.required_attrs) - set(kwargs)
        if required_not_in_kwargs:
            msg = 'Required arguments for Resource missed: {0}'.format(
                ', '.join(required_not_in_kwargs))
            raise Exception(msg)

        # set given params into attrs
        for key, val in kwargs.items():
            if key not in self.attrs:
                msg = 'Incorrect argument for Resource: {0}'.format(key)
                raise Exception(msg)

            setattr(self, key, val)

        # fill optionals
        optional_not_set = set(self.optional_attrs) - set(kwargs)
        for attr in self.optional_attrs:
            if attr in optional_not_set:
                value_generator = getattr(self, 'build_' + attr)
                setattr(self, attr, value_generator())

    def __str__(self):
        return str('Resource: ' + self.name)

    __repr__ = __str__

    def build_plural(self):
        return self.name + 's'

    def build_verbose(self):
        return self.plural.capitalize()

    def build_list_order(self):
        already_registered = registry.by_name.values()
        if not already_registered:
            return 0
        else:
            max_val = max(res.list_order for res in already_registered)
            return max_val + 1

    def build_fieldset_cls(self):
        return fields.Fieldset

    def build_default_filters(self):
        return {}

    def build_indexes_hint(self):
        return None

    def build_ignore_fields(self):
        return set()

    @property
    def schema(self):
        return SCHEMAS[self.name]


def get_resource_or_404(view_func):
    from django.http import Http404

    @wraps(view_func)
    def wrapper(request, resource, *args, **kwargs):
        found_by_name = registry.by_name.get(resource)
        found_by_plural = registry.by_plural.get(resource)
        resource_instance = found_by_name or found_by_plural

        if resource_instance is None:
            raise Http404('Resource `{0}` was not found'.format(resource))

        return view_func(request, resource_instance, *args, **kwargs)

    return wrapper


