# coding: utf-8
import json

import jsonschema

from django.contrib.staticfiles.finders import find as find_static

from collections import OrderedDict, defaultdict


class Registry(dict):
    def register(self, *schemas):
        for schema in schemas:
            self[schema.id] = schema


class Schema(object):
    def __init__(self, id):
        self.id = id
        self._preresolved = None
        self._schema = None
        self._flat = None
        self._hierarchical_index = None
        self._full_hierarchical_index = None
        self._flat_leaf = None

    def _get_static_filename(self):
        raise NotImplementedError

    def _get_filename(self):
        return find_static(self._get_static_filename())

    @property
    def schema(self):
        if self._schema is None:
            self._schema = get_schema(self._get_filename())

        return self._schema

    @property
    def flat(self):
        if self._flat is None:
            self._flat = get_schema(
                filename=self._get_filename(),
                preresolve=True,
                flatten=True,
            )

        return self._flat

    @property
    def flat_leaf(self):
        if self._flat_leaf is None:
            self._flat_leaf = OrderedDict((f, s) for f, s in self.flat.items() if s['is_leaf'])
        return self._flat_leaf

    @property
    def hierarchical_index(self):
        if self._hierarchical_index is None:
            self._hierarchical_index = defaultdict(dict)
            for field, schema in self.flat_leaf.items():
                field_path = ''
                for item in field.split('.'):
                    field_path = '%s.%s' % (field_path, item) if field_path else item
                    self._hierarchical_index[field_path][field] = schema

        return self._hierarchical_index

    @property
    def full_hierarchical_index(self):
        if self._full_hierarchical_index is None:
            self._full_hierarchical_index = defaultdict(set)
            for field, schema in self.flat.items():
                chunks = field.split('.')
                for n in range(len(chunks)):
                    key = '.'.join(chunks[:n + 1])
                    self._full_hierarchical_index[key].add(field)
        return self._full_hierarchical_index

    @property
    def preresolved(self):
        if self._preresolved is None:
            self._preresolved = get_schema(
                filename=self._get_filename(),
                preresolve=True,
            )

        return self._preresolved

    def get_field_schema(self, name):
        return self.flat.get(name)

    def validate_field(self, name, value):
        return jsonschema.validate(
            instance=value,
            schema=self.flat[name],
        )

    def validate(self, obj):
        return jsonschema.validate(
            instance=obj,
            schema=self.schema,
            resolver=_get_resolver_for_schema(self.schema),
        )


def get_schema(filename, preresolve=False, flatten=False):
    with open(filename) as inp:
        schema = json.load(inp)

    if preresolve:
        schema = _resolve(schema)

        if flatten:
            schema = dict(_flatten_schema(schema, skip_self=True))
            schema = OrderedDict(sorted(schema.items(), key=lambda f: f[0]))

    return schema


def _is_object(type_):
    return type_ == 'object' or isinstance(type_, list) and 'object' in type_


def _resolve(schema, resolver=None):
    if 'oneOf' in schema and len(schema) == 1:
        schema['oneOf'] = [_resolve_one(s, resolver) for s in schema['oneOf']]
    else:
        schema = _resolve_one(schema, resolver)

    return schema


def _resolve_one(schema, resolver=None):
    if resolver is None:
        resolver = _get_resolver_for_schema(schema)

    if '$ref' in schema:
        with resolver.resolving(schema['$ref']) as nested:
            schema = _resolve(nested, resolver)

    type_ = schema.get('type')

    if _is_object(type_):
        props = schema.get('properties', {})

        for name in props:
            props[name] = _resolve(props[name], resolver)
    elif type_ == 'array':
        schema['items'] = _resolve(schema['items'], resolver)

    return schema


def _flatten_schema(schema, prefix='', skip_self=False):

    if 'oneOf' in schema and len(schema) == 1:
        schemas = schema['oneOf']
    else:
        schemas = [schema]

    for schema in schemas:
        schema = schema.copy()
        type_ = schema.get('type')
        schema['is_leaf'] = not (_is_object(type_) or type_ == 'array')

        if not (skip_self and _is_object(type_)):
            yield prefix[:-1], schema

        if _is_object(type_):
            for name, value in sorted(schema.get('properties', {}).items()):
                for nested_name, nested_value in _flatten_schema(
                        value, '%s%s.' % (prefix, name)):
                    yield nested_name, nested_value
        elif type_ == 'array':
            for nested_name, nested_value in _flatten_schema(
                    schema['items'], prefix, skip_self=True):
                yield nested_name, nested_value


def _get_resolver_for_schema(schema):
    return jsonschema.RefResolver.from_schema(
        schema=schema,
        store=dict(
            ('%s.json' % s.id, s.schema)
            for s in SCHEMAS.values()
        ),
    )


SCHEMAS = Registry()


