# coding: utf-8
from __future__ import absolute_import

import time
import datetime
import hashlib

from django.conf import settings


class StateManager(object):
    def __init__(self, storage_manager, read_only=False):
        self.storage_manager = storage_manager
        self.read_only = read_only

    @property
    def collection(self):
        return self.storage_manager.get_meta_namespace(read_only=self.read_only)['state']

    def get_state_history(self, limit=None):
        states = self.collection.get(
            {'api_version': settings.API_VERSION},
            sort='-started_at',
            limit=limit,
        )

        if not states:
            states = [self._create_state('update', '', '')]

        return states

    def get_state(self):
        return self.get_state_history(limit=1)[0]

    def set_state(self, name):
        assert(name in ('init', 'update'))

        current = self.get_state()

        if current['name'] == 'init' and name == 'update':
            read = current['prefixes']['write']
        else:
            read = current['prefixes']['read']

        if name == 'init':
            write = '%s_%s_' % (
                settings.STATIC_API_NAMES_PREFIX,
                hashlib.md5(str(time.time())).hexdigest()[:7],
            )
        else:
            write = current['prefixes']['write']

        state = self._create_state(name, read, write)

        self.collection.put({}, state)

        return state

    def compact(self):
        current = self.get_state()

        states = self.get_state_history() + [self._create_state('init', '', '')]

        for state in states:
            prefix = state['prefixes']['write']

            if state['name'] == 'init' and prefix not in current['prefixes'].values():
                namespace = self.storage_manager.get_entity_namespace(prefix)

                namespace.drop()

    def _create_state(self, name, read, write):
        return {
            'name': name,
            'api_version': settings.API_VERSION,
            'prefixes': {'read': read, 'write': write},
            'started_at': datetime.datetime.now(),
        }
