# coding: utf-8
from __future__ import unicode_literals

import logging
from datetime import datetime, timedelta

from pymongo import MongoClient

from django.conf import settings
from static_api import state

from . import namespace, exceptions


log = logging.getLogger(__name__)


class StorageManager(object):
    def __init__(self):
        self._db = None
        self._cached_read_namespace = None
        self._cached_read_namespace_datetime = None

        # http://docs.mongodb.org/manual/reference/limits/#Number-of-Indexes-per-Collection
        for collection, indexes in settings.API_INDEXES.items():
            if len(indexes) > 64:
                raise ValueError('To many indexes (%s out of 64) for'
                                 ' collection: %s' % (len(indexes), collection))

    def _init(self, db_name=None, force=False):
        if self._db is None or force:
            with exceptions.wrapper:
                self._connection = MongoClient(
                    settings.MONGO_URL,
                    connectTimeoutMS=settings.STATIC_API_MONGO_CONNECT_TIMEOUT,
                    socketTimeoutMS=settings.STATIC_API_MONGO_SOCKET_TIMEOUT,
                    maxPoolSize=settings.STATIC_API_MONGO_MAX_POOL_SIZE,
                    replicaSet=settings.MONGO_REPLICA_SET,
                )
                self._db = self._connection[db_name or settings.MONGO_DB]

    @property
    def db(self):
        self._init()

        return self._db

    def get_info(self):
        return {
            'replica_set': {},
            'database': {},
        }

    def get_collections_info(self):
        with exceptions.wrapper:
            for collection in self.db.collection_names(include_system_collections=False):
                yield collection, self.db.command('collStats', collection, scale=1024 ** 2)

    def get_entity_namespace(self, prefix, state_name=None, read_only=False):
        return namespace.CollectionNamespace(
            storage_manager=self,
            prefix=prefix,
            available=settings.STATIC_API_ENTITY_COLLECTIONS,
            state_name=state_name,
            read_only=read_only,
            indexes=settings.API_INDEXES,
        )

    def get_read_namespace(self):
        now = datetime.now()
        expire_time = timedelta(seconds=settings.STATIC_API_READ_NAMESPACE_CACHE_SECONDS)
        if self._cached_read_namespace is None or self._cached_read_namespace_datetime + expire_time < now:
            state = self.get_state_manager(True).get_state()
            result = self.get_entity_namespace(
                prefix=state['prefixes']['read'],
                state_name=state['name'],
                read_only=True
            )

            self._cached_read_namespace = result
            self._cached_read_namespace_datetime = now

        return self._cached_read_namespace

    def get_write_namespace(self):
        state = self.get_state_manager().get_state()

        return self.get_entity_namespace(
            prefix=state['prefixes']['write'],
            state_name=state['name'],
        )

    def get_meta_namespace(self, read_only=False):
        return namespace.CollectionNamespace(self, '',
                                             settings.STATIC_API_META_COLLECTIONS,
                                             read_only=read_only)

    def is_read_equals_write(self):
        state = self.get_state_manager().get_state()

        return state['prefixes']['read'] == state['prefixes']['write']

    def reset(self, include_meta=False):
        self.get_write_namespace().reset()

        if include_meta:
            self.get_meta_namespace().reset()

        self.ensure_indexes()

    def ensure_indexes(self):
        self.get_write_namespace().ensure_indexes()
        self.get_meta_namespace().ensure_indexes()
        self.ensure_misc_indexes()

    def ensure_misc_indexes(self):
        for collection, index in settings.MISC_INDEXES.items():
            self.db[collection].ensure_index(index['keys'], name=index['name'], unique=index['unique'])

    def get_state_manager(self, read_only=False):
        return state.StateManager(self, read_only)


manager = StorageManager()
