# coding: utf-8
from __future__ import unicode_literals

import logging
from datetime import datetime
from contextlib import contextmanager

import dpath.util
from pymongo import operations as ops

from django.conf import settings


from . import utils, exceptions, filter_arrays_helper


log = logging.getLogger(__name__)


@contextmanager
def log_elapsed_time(log_prefix):
    start = datetime.now()
    try:
        yield
    finally:
        elapsed = datetime.now() - start
        log.debug('%s: elapsed %s ms', log_prefix, elapsed.total_seconds() * 1000)


class Collection(object):
    def __init__(self, collection, context=None, read_only=False):
        self._collection = collection
        self._context = context
        self._read_only = read_only

    def count(self):
        return self._collection.count_documents({}, hint='_id_')

    def estimated_document_count(self):
        return self._collection.estimated_document_count()

    def get(self, lookup, fields=None, sort=None, limit=None):
        log.debug(
            'Call `%s.get` with lookup: %s',
            self._collection.name, lookup
        )

        docs = self._find(lookup, fields)

        if sort:
            docs = docs.sort(utils._parse_sort(sort))

        if limit:
            docs = docs.limit(limit)

        return list(docs)

    def get_one(self, lookup, fields=None, sort=None, hint=None):
        log.debug(
            'Call `%s.get_one` with lookup: %s',
            self._collection.name, lookup
        )

        sort = utils._parse_sort(sort) if sort else None
        with log_elapsed_time('_find_one'):
            return self._find_one(lookup, fields, sort=sort, hint=hint)

    def explain(self, lookup, page, limit=settings.PAGE_SIZE, fields=None, sort=None, hint=None):
        log.debug(
            'Call `%s.explain` with lookup: %s; page: %s; sort: %s; limit: %s; fields %s; hint: %s',
            self._collection.name, lookup, page, sort, limit, fields, hint
        )

        skip = limit * (page - 1)
        docs = self._find(lookup, fields, batch_size=limit, hint=hint).skip(skip).limit(limit)

        if sort:
            docs = docs.sort(utils._parse_sort(sort))

        result = docs.explain()
        sig_hash = result['$clusterTime']['signature']['hash']
        result['$clusterTime']['signature']['hash'] = ','.join('0x{:02x}'.format(ord(x)) for x in sig_hash)
        return result

    def get_page(self, lookup, page, limit=settings.PAGE_SIZE, fields=None, sort=None, hint=None, nototal=False):
        log.debug(
            'Call `%s.get_page` with lookup: %s; page: %s; sort: %s; limit: %s; fields %s; hint: %s',
            self._collection.name, lookup, page, sort, limit, fields, hint
        )

        with log_elapsed_time('_count'):
            total = None if nototal else self._count(lookup, hint=hint)

        skip = limit * (page - 1)
        docs = (
            self
            ._find(lookup, fields, batch_size=limit, hint=hint)
            .skip(skip)
            .limit(limit)
            .max_time_ms(settings.STATIC_API_QUERY_TIMEOUT)
        )

        if sort:
            docs = docs.sort(utils._parse_sort(sort))

        with log_elapsed_time('_find'):
            result = list(docs)

        return result, total

    def put(self, lookup, data, raw=False, merge=False):
        assert(not self._read_only)

        log.debug(
            'Call `%s.put` with lookup: %s',
            self._collection.name, lookup
        )

        if raw or not lookup:
            doc = data
        else:
            if merge:
                data = utils.flatten_dict(data)

            doc = {'$set': data}

        self._replace(lookup, doc, upsert=True)

    def delete(self, lookup):
        assert(not self._read_only)

        log.debug(
            'Call `%s.delete` with lookup: %s',
            self._collection.name, lookup
        )

        with exceptions.wrapper:
            self._collection.remove(lookup, w=1, multi=True)

    def put_nested(self, attr, lookup, data, parent_lookup=None):
        assert(not self._read_only)

        log.debug(
            'Call `%s.put_nested` to `%s` with lookup: %s',
            self._collection.name, attr, lookup
        )

        if '$' in attr:
            if attr.count('$') > 1:
                self._put_to_array_manual(attr, data, lookup, parent_lookup)
            else:
                self._put_to_array(attr, data, lookup, parent_lookup)
        else:
            q = {}
            q.update(lookup)
            if parent_lookup:
                q.update(parent_lookup)

            write_result = self._update(q, {'$set': {attr: data}})

            # https://github.com/mongodb/mongo-python-driver/blob/de1e29305c70a309aa52d30b780b3eb332265ab2/pymongo/results.py#L117
            if write_result.get('n', 0) > 0:
                log.debug('Lookup exists: %s', q)
            elif parent_lookup:
                log.debug(
                    'Lookup doesn\'t exist: %s. Assign to parent: %s',
                    q, parent_lookup
                )

                self._update(parent_lookup, {'$set': {attr: data}})
            else:
                log.debug('Lookup doesn\'t exist: %s', q)

    def update_nested(self, path, element_id, data):
        if '$' in path:
            raise ValueError('use update_nested_*_array functions to update elements of arrays')

        documents_filter = {
            path + '.id': element_id,
        }

        update = {'$set': {path: data}}
        return ops.UpdateMany(
            filter=documents_filter,
            update=update,
            upsert=False,
        )

    def update_nested_array_element(self, path, element_id, data=None, delete=False):
        if data and delete:
            raise ValueError('''you can't provide data while deleting document''')

        if not data and not delete:
            raise ValueError('''you have to provide data while updating''')

        if not path.endswith('.$'):
            raise ValueError('path should ends with .$ as you trying to update entire array element')

        def make_array_filters(path_elems, elem_id):
            array_filters = []
            for idx, path_elem in enumerate(path_elems[1:]):
                array_filters.append(
                    filter_arrays_helper.make_filter_that_path_element_exists(idx, path_elem)
                )

            if not delete:
                array_filters.append({'elem.id': elem_id})
            return array_filters

        def make_set_path(path_elems):
            set_path = filter_arrays_helper.make_set_path_elements_without_last(path_elems)
            if not delete:
                set_path.append('$[elem]')

            return '.'.join(set_path)

        path = path[0:-len('.$')]
        path_elems = filter_arrays_helper.parse_path_elements(path)
        array_filters = make_array_filters(path_elems, element_id)
        documents_filter = filter_arrays_helper.make_document_filter(path, element_id)

        update = (
            {'$pull': {make_set_path(path_elems): {'id': element_id}}}
            if delete
            else {'$set': {make_set_path(path_elems): data}}
        )

        return ops.UpdateMany(
            filter=documents_filter,
            update=update,
            array_filters=array_filters,
            upsert=False,
        )

    def update_in_nested_array_element(self, path, element_id, data):
        if path.endswith('.$'):
            raise ValueError('''path shouldn't ends with .$ as you trying to update contents of array element''')

        filter_arrays_helper.raise_value_error_for_too_simple_case(path)

        def make_array_filters(path_elems, elem_id):
            array_filters = []
            for idx, path_elem in enumerate(path_elems[1:-1]):
                array_filters.append(
                    filter_arrays_helper.make_filter_that_path_element_exists(idx, path_elem)
                )

            array_filters.append(
                filter_arrays_helper.make_last_element_filter(path_elems[-1], elem_id)
            )

            return array_filters

        def make_set_path(path_elems):
            set_path = filter_arrays_helper.make_set_path_elements_without_last(path_elems[:-1])
            set_path.append('$[elem]')
            set_path.append(path_elems[-1])
            return '.'.join(set_path)

        path_elems = filter_arrays_helper.parse_path_elements(path)
        array_filters = make_array_filters(path_elems, element_id)
        documents_filter = filter_arrays_helper.make_document_filter(path, element_id)
        update = {'$set': {make_set_path(path_elems): data}}

        return ops.UpdateMany(
            filter=documents_filter,
            update=update,
            array_filters=array_filters,
            upsert=False,
        )

    def execute_bulk(self, updates, ordered=True):
        self._collection.bulk_write(updates, ordered)

    def delete_nested(self, attr, lookup, parent_lookup=None, set_none=False):
        assert(not self._read_only)

        log.debug(
            'Call `%s.delete_nested` to `%s` with lookup: %s',
            self._collection.name, attr, lookup
        )

        if '$' in attr:
            if attr.count('$') > 1 or not attr.endswith('.$'):
                self._delete_from_array_manual(attr, lookup, parent_lookup,
                                               set_none)
            else:
                self._delete_from_array(lookup, parent_lookup)
        else:
            if set_none:
                placeholder = None
            else:
                placeholder = self._get_nested_deleted_placeholder(attr, lookup)

            q = {}
            q.update(lookup)
            if parent_lookup:
                q.update(parent_lookup)

            self._update(q, {'$set': {attr: placeholder}})

    # Приватный интерфейс

    def _find(self, *args, **kwargs):
        """Возвращает курсор по параметрам
        """
        if 'hint' in kwargs and kwargs['hint'] is None:
            del kwargs['hint']

        with exceptions.wrapper:
            return self._collection.find(*args, **kwargs)

    def _count(self, lookup, hint=None, *args, **kwargs):
        if not lookup and not hint:
            hint = '_id_'  # Use default collection index for counting without filters

        if hint:
            kwargs['hint'] = hint

        kwargs['maxTimeMS'] = settings.STATIC_API_QUERY_TIMEOUT

        with exceptions.wrapper:
            return self._collection.count_documents(lookup, *args, **kwargs)

    def _find_one(self, *args, **kwargs):
        """Возврашает один документ или None по параметрам
        """
        if 'hint' in kwargs and kwargs['hint'] is None:
            del kwargs['hint']

        with exceptions.wrapper:
            return self._collection.find_one(*args, **kwargs)

    def _put_to_array(self, attr, data, lookup, parent_lookup=None):
        q = {}
        q.update(lookup)
        if parent_lookup:
            q.update(parent_lookup)

        q = utils._strip_positional_operator(q)

        write_result = self._update(q, {'$set': {attr: data}})

        # https://github.com/mongodb/mongo-python-driver/blob/de1e29305c70a309aa52d30b780b3eb332265ab2/pymongo/results.py#L117
        if write_result.get('n', 0) > 0:
            log.debug('Lookup exists: %s', q)
        elif parent_lookup:
            log.debug(
                'Lookup doesn\'t exist: %s. Assign to parent: %s',
                q, parent_lookup
            )

            self._update(parent_lookup, {'$push': {attr.rstrip('.$'): data}})
        else:
            log.debug('Lookup doesn\'t exist: %s', q)

    def _put_to_array_manual(self, attr, data, lookup, parent_lookup=None):
        q = {}
        q.update(lookup)
        if parent_lookup:
            q.update(parent_lookup)

        q = utils._strip_positional_operator(q)

        with exceptions.wrapper:
            field = lookup.keys()[0].replace('.$', '')
            docs = self._collection.find(q, {'_id': 1, field: 1})

        updates = []
        for doc in docs:
            resolved_path = utils._find_in_dict(doc, lookup)

            path = utils._apply_resolved_path(attr, resolved_path)

            log.debug('Assign to attr: %s', path)

            updates.append(ops.UpdateOne({'_id': doc['_id']}, {'$set': {path: data}}))

        if updates:
            log.debug('Lookup exists: %s', q)
            self._collection.bulk_write(updates, ordered=False)
            return

        if not parent_lookup:
            log.debug('Lookup doesn\'t exist: %s', q)
            return

        log.debug(
            'Lookup doesn\'t exist: %s. Assign to parent: %s',
            q, parent_lookup
        )

        lookup = utils._strip_positional_operator(parent_lookup)

        with exceptions.wrapper:
            field = lookup.keys()[0]
            docs = self._collection.find(lookup, {'_id': 1, field: 1})

        for doc in docs:
            resolved_path = utils._find_in_dict(doc, parent_lookup)
            path = utils._apply_resolved_path(attr.rstrip('.$'), resolved_path)
            updates.append(ops.UpdateOne({'_id': doc['_id']}, {'$push': {path: data}}))

        if updates:
            self._collection.bulk_write(updates, ordered=False)

    def _delete_from_array(self, lookup, parent_lookup=None):
        assert(len(lookup) == 1)

        path, value = lookup.items()[0]

        array, tail = [bit.strip('.') for bit in path.split('$', )]

        q = {}
        q.update(lookup)
        if parent_lookup:
            q.update(parent_lookup)
        q = utils._strip_positional_operator(q)

        self._update(q, {'$pull': {array: {tail: value}}})

    def _delete_from_array_manual(self, attr, lookup, parent_lookup=None,
                                  set_none=False):
        q = {}
        q.update(lookup)
        if parent_lookup:
            q.update(parent_lookup)

        q = utils._strip_positional_operator(q)

        with exceptions.wrapper:
            field = lookup.keys()[0].replace('.$', '')
            docs = self._collection.find(q, {'_id': 1, field: 1})

        updates = []
        for doc in docs:
            resolved_path = utils._find_in_dict(doc, lookup)

            path = utils._apply_resolved_path(attr, resolved_path)

            if attr.endswith('.$'):
                array_path, index = path.rsplit('.', 1)
                array_lookup = {k.rsplit('$.', 1)[1]: v for k, v in lookup.items()}
                updates.append(ops.UpdateOne({'_id': doc['_id']}, {'$pull': {array_path: array_lookup}}))
            else:
                if set_none:
                    placeholder = None
                else:
                    placeholder = self._get_nested_deleted_placeholder(attr, lookup)
                updates.append(ops.UpdateOne({'_id': doc['_id']}, {'$set': {path: placeholder}}))

        if updates:
            self._collection.bulk_write(updates, ordered=False)

    def _replace(self, lookup, doc, upsert=False):
        if not lookup:
            with exceptions.wrapper:
                self._collection.insert(doc, w=1)
        else:
            with exceptions.wrapper:
                self._collection.update(lookup, doc, upsert=upsert, w=1)

    def _update(self, lookup, doc):
        if self._context:
            defaults = {'$set': utils.flatten_dict(self._context)}

            dpath.util.merge(defaults, doc)

            doc = defaults

        with exceptions.wrapper:
            return self._collection.update(lookup, doc, w=1, multi=True)

    def _get_nested_deleted_placeholder(self, attr, lookup):
        id_ = lookup.get(attr + '.id')

        return dict(settings.STATIC_API_DELETED_NESTED_PLACEHOLDER_EXTRA,
                    **{'id': id_})
