# coding: utf-8
from __future__ import unicode_literals

import logging
from pymongo.read_preferences import ReadPreference
from contextlib import contextmanager

from . import collection, exceptions


log = logging.getLogger(__name__)


class CollectionNamespace(object):
    def __init__(self, storage_manager, prefix, available, state_name=None, read_only=False,
                 indexes=None):
        self.storage_manager = storage_manager
        self.prefix = prefix
        self.available = available
        self.context_stack = []
        self.state_name = state_name
        self.read_only = read_only
        self.indexes = indexes or {}

    @property
    def db(self):
        return self.storage_manager.db

    @contextmanager
    def context(self, data):
        self.context_stack.append(data)

        try:
            yield self
        finally:
            self.context_stack.pop()

    def __getattr__(self, name):
        return self[name]

    def __getitem__(self, name):
        try:
            context = self.context_stack[-1]
        except IndexError:
            context = None

        if self.read_only:
            read_preference = ReadPreference.SECONDARY_PREFERRED
        else:
            read_preference = ReadPreference.PRIMARY

        coll = self._get_collection(name).with_options(
            read_preference=read_preference,
        )

        return collection.Collection(coll, context, self.read_only)

    def __iter__(self):
        for name in self.available:
            yield name, self[name]

    def drop(self):
        for name in self.available:
            with exceptions.wrapper:
                self.db.drop_collection(self._get_full_name(name))

    def reset(self):
        self.drop()

        self.ensure_indexes()

    def ensure_indexes(self):
        for name in self.available:
            try:
                indexes = self.indexes[name]
            except KeyError:
                continue

            collection = self._get_collection(name)

            for index in indexes:
                with exceptions.wrapper:
                    collection.ensure_index(index)

    def _get_full_name(self, name):
        if name in self.available:

            return self.prefix + name

        raise ValueError('Unknown collection: %s' % name)

    def _get_collection(self, name):
        with exceptions.wrapper:
            return self.db[self._get_full_name(name)]
