# coding: utf-8
from __future__ import unicode_literals

import logging
import functools

from django.conf import settings

from celery import shared_task
from static_api import (
    listeners,
    helpers,
    metrics,
    client,
    utils,
    storage,
    message,
)
from static_api.lock import locked


log = logging.getLogger(__name__)


def _process_message(raw_message, namespace):
    id_ = raw_message['id']
    creation_time = raw_message['creation_time']

    listeners.registry.dispatch(client.DjangoMessage(raw_message), namespace)

    message.manager.set_last_message(id_, creation_time)


def retry_storage_errors(func):
    @functools.wraps(func)
    def _wrapper(self, *args, **kwargs):
        try:
            return func(self, *args, **kwargs)
        except storage.exceptions.StorageError as e:
            log.exception('Storage error: %s. Retrying `%s`', e, func.__name__)

            raise self.retry(countdown=2 ** self.request.retries, exc=True)

    return _wrapper


@shared_task(bind=True, max_retries=None)
@locked('process')
def reset(self, include_meta=False):
    storage.manager.reset(include_meta=include_meta)


@shared_task(bind=True)
@locked('process')
def update(self):
    current_state = storage.manager.get_state_manager().get_state()
    if current_state['name'] != 'update':
        return

    namespace = storage.manager.get_write_namespace()
    manager = message.manager

    if settings.USE_NEW_UPDATE_LOGIC:
        handled_messages_ids = set()
        for raw_message in manager.get_unhandled_messages_new(settings.UPDATE_LIMIT):
            message_id = raw_message['id']
            log.info('Handling message %s', message_id)
            _process_message(raw_message, namespace)
            handled_messages_ids.add(message_id)

        if not handled_messages_ids:
            return

        log.info('Handled all unsent messages, mark them sent')
        manager.mark_messages_as_sent_new(handled_messages_ids)
    else:
        for message_range in manager.get_unhandled_messages(settings.UPDATE_LIMIT):
            log.info('Handling range: %r', message_range)

            for raw_message in message_range:
                _process_message(raw_message, namespace)


@shared_task(bind=True)
@locked('process')
@retry_storage_errors
def init(self, entity=None):
    message.manager.trim_ids()

    meta_data = helpers.get_meta()

    entities = [entity] if entity else meta_data['entities']

    namespace = storage.manager.get_write_namespace()

    for entity in entities:
        for raw_message in helpers.get_entity_data(entity):
            _process_message(raw_message, namespace)

    last_message = meta_data['last_message']
    message.manager.set_last_message(
        id_=last_message['id'],
        creation_time=last_message['creation_time'],
        trim=True
    )

    storage.manager.get_state_manager().set_state('update')
    listeners.registry.dispatch_init_hooks(entities, namespace)


@shared_task(bind=True, max_retries=None)
@locked('graphite')
def graphite(self):
    deltas = helpers.get_current_master_deltas()

    metrics.send_graphite_metrics({
        utils.get_metric_path('delta_ids'): deltas['delta_ids'],
        utils.get_metric_path('delta_time'): deltas['delta_time'],
    })


@shared_task(max_retries=None)
def ensure_indexes():
    storage.manager.ensure_indexes()


@shared_task(max_retries=None)
def compact():
    message.manager.compact()

    storage.manager.get_state_manager().compact()
