# coding: utf-8

import dateutil.parser
import pytz
import yenv

from django.conf import settings
from django.core.cache import cache


def split_str(text):
    return [v.strip() for v in text.split(u',')]


def parse_time(dt, strip_microseconds=False):
    parsed = dateutil.parser.parse(dt)

    if parsed.tzinfo is None:
        parsed = pytz.timezone('Europe/Moscow').localize(parsed)

    if strip_microseconds:
        parsed = parsed.replace(microsecond=0)

    return parsed.astimezone(pytz.utc)


def isoformat_with_millis(dt):
    assert dt.tzinfo
    result = dt.strftime('%Y-%m-%dT%H:%M:%S.%f%z')
    result = result[:-2] + ':' + result[-2:]
    return result


def get_metric_path(name):
    prefix = settings.STATIC_API_GRAPHITE_METRIC_PREFIX

    if prefix:
        return '.'.join((
            prefix,
            yenv.type,
            name,
            settings.API_VERSION.replace('.', '_'),
        ))


def cached_function(cache_time=60):
    def deco(func):
        def inner(*args, **kwargs):
            cache_key = '{module}.{name}.{args}.{kwargs}'.format(
                module=func.__module__,
                name=func.__name__,
                args=str(args),
                kwargs=str(sorted(kwargs.items())),
            )
            value = cache.get(cache_key)
            if value is not None:
                return value

            result = func(*args, **kwargs)
            cache.set(cache_key, result, cache_time)
            return result
        return inner
    return deco
