# coding: utf-8
from __future__ import unicode_literals

import logging
import math

from django.conf import settings
from pymongo import ReadPreference
from pymongo.errors import ExecutionTimeout
from static_api import (
    request_parsers as parsers,
    resources,
    storage,
    utils as static_utils,
)

from . import utils
from .docs import doc


MAX_HEADER_LENGTH = 15500
INFINITY_BATCH_SIZE = 10**9  # mongo limits batch size to 4-16MB

log = logging.getLogger(__name__)


def _get_request_data(request):
    request_data = {}

    if request.method == 'POST':
        request_data.update(request.POST.dict())

    request_data.update(request.GET.dict())

    return request_data


def get_nested_fields(fields, resource):
    result = set()
    full_hierarchical_index = resource.schema.full_hierarchical_index
    for field in fields:
        result.update(set(full_hierarchical_index.get(field, [])))

    return result


@static_utils.cached_function(cache_time=60 * 3)
def _get_fields_access_from_mongo(subject_type, subject_id, resource):
    fields_access_collection = storage.manager.db.get_collection(
        name=settings.STATIC_API_FIELD_ACCESS_RULES_COLLECTION,
        read_preference=ReadPreference.SECONDARY_PREFERRED,
    )
    rows = fields_access_collection.find(
        filter={
            'subject_type': subject_type,
            'subject_id': str(subject_id),
            'resource': resource.name
        },
        batch_size=INFINITY_BATCH_SIZE,
    )
    fields = {row.get('field') for row in rows}

    if settings.STATIC_API_WILDCARD_FIELD_ACCESS in fields:
        return {settings.STATIC_API_WILDCARD_FIELD_ACCESS}
    else:
        return fields


def _get_permitted_fields(subject, resource):
    if subject:
        if subject.service_ticket:
            subject_id = subject.service_ticket.src
            subject_type = 'tvm_app'
        else:
            subject_id = subject.uid
            subject_type = 'user'
        mongo_fields = _get_fields_access_from_mongo(subject_type, subject_id, resource)
        with_nested_fields = mongo_fields | get_nested_fields(mongo_fields, resource)
        return with_nested_fields
    return set()


@resources.get_resource_or_404
def resource_view(request, resource, **override_params):
    request_data = _get_request_data(request)
    request_data.update(override_params)

    if request_data.get('_doc') and request_data['_doc'] == '1':
        return doc(request, resource.name)

    yauser = getattr(request, 'yauser', None)

    if settings.STATIC_API_CHECK_FIELDS_ACCESS:
        permitted_fields = _get_permitted_fields(yauser, resource)
    else:
        permitted_fields = {settings.STATIC_API_WILDCARD_FIELD_ACCESS}

    query_params_parser = parsers.SpecialParamsParser(resource, permitted_fields, request_data)
    if not query_params_parser.is_valid():
        kwargs = {}
        if query_params_parser.has_forbidden_fields:
            status_code = 403
            error_message = 'Fields access forbidden. Refer to %s' % settings.STATIC_API_FIELDS_FORBIDDEN_URL
            kwargs.update(role_request_url=settings.IDM_ROLE_REQUEST_URL_TEMPLATE % {'resource': resource.name})
        else:
            status_code = 400
            error_message = 'Parameters validation error'

        return utils.build_error_json_response(
            msg=error_message,
            details=query_params_parser.errors,
            status_code=status_code,
            **kwargs
        )
    query_params = query_params_parser.cleaned_data_obj
    return search(request, resource, query_params)


def resource_detail_view(request, resource, id, **kwargs):
    return resource_view(request, resource, id=id, _one='1', **kwargs)


def _get_namespace(params):
    manager = storage.manager
    if params.write:
        namespace = manager.get_write_namespace()
    else:
        namespace = manager.get_read_namespace()
    return namespace


def search(request, resource, query_params):
    try:
        namespace = _get_namespace(query_params)
        coll = namespace[resource.name]
        links = {}
        if query_params.debug:
            links.update(utils.make_production_link(request, query_params.fields))

        if query_params.explain:
            data = coll.explain(
                query_params.query,
                page=query_params.page,
                limit=query_params.limit,
                fields=query_params.fields,
                sort=query_params.sort or settings.STATIC_API_DEFAULT_SORT,
                hint=query_params.hint,
            )
        elif query_params.one:
            data = coll.get_one(
                query_params.query,
                fields=query_params.fields,
                sort=query_params.sort if query_params.sort else None,
                hint=query_params.hint,
            )

            if not data:
                return utils.build_error_json_response(
                    msg="not found",
                    details={"request": _get_request_data(request)},
                    status_code=404,
                    pretty=query_params.pretty,
                )

        else:
            rows, total = coll.get_page(
                query_params.query,
                page=query_params.page,
                limit=query_params.limit,
                fields=query_params.fields,
                sort=query_params.sort or settings.STATIC_API_DEFAULT_SORT,
                hint=query_params.hint,
                nototal=query_params.nopage,
            )

            if query_params.nopage:
                data = {
                    'result': rows,
                    'page': query_params.page,
                    'limit': query_params.limit,
                }
            else:
                pages = int(math.ceil(float(total) / query_params.limit))
                links.update(utils.make_links(request, query_params.page, pages))

                data = {
                    'result': rows,
                    'total': total,
                    'page': query_params.page,
                    'limit': query_params.limit,
                    'pages': pages,
                    'links': dict(links),
                }

        response = utils.build_json_response(data, pretty=query_params.pretty)

        if links:
            link_header = ', '.join(
                '<%s>; rel="%s"' % (link, name)
                for name, link in links.items()
            )
            if len(link_header) <= MAX_HEADER_LENGTH:
                response['Link'] = link_header

        return response
    except ExecutionTimeout:
        return utils.build_error_json_response(
            msg='Your request took too long time and was aborted',
            details=(
                'Change your request. Read docs on https://staff-api.yandex-team.ru/v3/'
                ' and https://wiki.yandex-team.ru/users/denis-p/Staff-API-Recipes/.'
                ' Ask help on https://st.yandex-team.ru/createTicket?queue=TOOLSUP'
                ' or https://abc.yandex-team.ru/services/staff/duty/'
            ),
            status_code=429,
            pretty=query_params.pretty,
        )

    except Exception as e:
        if settings.DEBUG:
            raise

        log.exception('Search failed:')

        return utils.build_error_json_response(
            msg=unicode(e),
            status_code=500,
            pretty=query_params.pretty,
        )
