# coding: utf-8
from __future__ import unicode_literals

import json
from datetime import datetime

from bson import ObjectId, Timestamp
from django import http
from query_exchange import process_query


class MongoDocumentEncoder(json.JSONEncoder):
    def default(self, o):
        if isinstance(o, datetime):
            return o.isoformat()
        if isinstance(o, Timestamp):
            return o.as_datetime().isoformat()
        elif isinstance(o, ObjectId):
            return str(o)
        return super(MongoDocumentEncoder, self).default(o)


def build_error_json_response(msg, details=None, status_code=400, pretty=True, **kwargs):
    data = dict(error_message=msg, **kwargs)
    if details is not None:
        data['details'] = details
    return build_json_response(
        data=data,
        status_code=status_code,
        pretty=pretty,
    )


def build_json_response(data, status_code=200, pretty=False):
    return http.HttpResponse(
        content=dump_response_data(data, pretty),
        status=status_code,
        content_type='application/json; charset=utf-8'
    )


def dump_response_data(data, pretty=False):
    return json.dumps(
        data,
        indent=2 if pretty else None,
        cls=MongoDocumentEncoder,
        ensure_ascii=False,
        encoding='utf-8',
    )


def make_links(request, page, pages):
    def _link(page_num):
        if page_num:
            query = process_query(request.GET, add=[('_page', page_num)])
        else:
            query = process_query(request.GET, exclude=('_page',))

        return request.build_absolute_uri(request.path + '?' + query)

    links = {}

    if page > 1:
        links['prev'] = _link(page - 1)
        links['first'] = _link(0)

    if page < pages:
        links['next'] = _link(page + 1)
        links['last'] = _link(pages)

    return list(links.items())


def make_production_link(request, fields):
    fields_str = ','.join(f for f, fs in fields.items() if fs)
    query = process_query(request.GET, add=[('_fields', fields_str)], exclude=('_debug',))
    link = request.build_absolute_uri(request.path + '?' + query)
    yield 'production_uri', link
