# coding: utf-8
from __future__ import unicode_literals, print_function, absolute_import, division

try:
    import uwsgi
except ImportError:
    uwsgi = None

import logging
import os
import re

from django.http import HttpResponse

from ylog.context import log_context

from competentum.log_formatter import DEPLOY_VARS

logger = logging.getLogger(__name__)


def ping_middleware(get_response):
    ping_re = re.compile(r'^/ping/?$')

    def middleware(request):
        if ping_re.match(request.path):
            return HttpResponse('OK')

        return get_response(request)

    return middleware


def request_id_middleware(get_response):
    def middleware(request):
        request.context = log_context(request_id=request.META.get('HTTP_X_REQ_ID', '')).__enter__()

        response = get_response(request)

        request.context.__exit__(None, None, None)

        return response

    return middleware


class UwsgiLogMiddleware(object):

    USER_IP_HEADERS = (
        'HTTP_X_CLIENT_REAL_IP',
        'HTTP_X_REAL_IP',
        'HTTP_X_FORWARDED_FOR',
        'REMOTE_ADDR',
    )

    HOST_HEADERS = (
        'HTTP_X_FORWARDED_HOST_STAFF',
        'HTTP_X_FORWARDED_HOST',
        'HTTP_HOST',
    )

    def extract_actual_header(self, request, headers):
        for header_name in headers:
            ip = request.META.get(header_name)
            if ip:
                return ip
        return None

    if uwsgi:
        def __init__(self, get_response):
            self.get_response = get_response

        def __call__(self, request):
            try:
                yandex_login = request.COOKIES.get('yandex_login')
                if yandex_login:
                    uwsgi.set_logvar('yandex_login', yandex_login)

                user_ip = self.extract_actual_header(request, self.USER_IP_HEADERS)
                if user_ip:
                    uwsgi.set_logvar('user_ip', user_ip)

                host = self.extract_actual_header(request, self.HOST_HEADERS)
                if host:
                    uwsgi.set_logvar('real_host', host)
            except Exception:
                logger.exception('Exception on set uWSGI logvar')

            self._set_deploy_vars()

            return self.get_response(request)

        def process_view(self, request, *args, **kwargs):
            try:
                uwsgi.set_logvar('view_name', request.resolver_match.view_name)
                #TODO: move session staff here
                user = getattr(request, 'user', None)
                if user:
                    uwsgi.set_logvar('yandex_login', user.username)

            except Exception:
                logger.exception('Exception on set uWSGI logvar')

        def _set_deploy_vars(self):
            for key in DEPLOY_VARS:
                uwsgi.set_logvar(key, os.environ[key])

    else:
        def __init__(self, get_response):
            self.get_response = get_response

        def __call__(self, request):
            return self.get_response(request)