# -*- coding: utf-8 -*-

from __future__ import unicode_literals, absolute_import

import csv
from logging import getLogger
from requests import get, Timeout
from competentum.report.person import Person
from competentum import settings

logger = getLogger(__name__)


def _process_department(result, departments_to_download, department):
    department_id, department_name, parent_department_id = department

    if department_id in result:
        return

    result[department_id] = (
        department_id,
        department_name,
        parent_department_id
    )

    if parent_department_id and parent_department_id not in result:
        departments_to_download.add(parent_department_id)

    if department_id in departments_to_download:
        departments_to_download.remove(department_id)


def get_departments(persons, header):
    result_departments = {}
    departments_to_download = set()

    for person in persons:
        assert isinstance(person, Person)
        department = (
            person.department_id,
            person.department_name,
            person.parent_department_id,
        )

        _process_department(result_departments, departments_to_download, department)

    url = settings.STAFF_API_HOST + (
        '/v3/groups?department.id={departments}'
        '&_fields=department.id,department.name,parent.department.id'
    )

    while departments_to_download:
        query = url.format(departments=','.join((str(d) for d in departments_to_download)))

        try:
            response = get(query, headers=header, timeout=5)

        except Timeout:
            logger.error(
                'Timeout on connect to the %s',
                url,
            )
            raise

        response = response.json()

        for item in response['result']:
            department = (
                item['department']['id'],
                item['department']['name']['full']['ru'],
                item['parent']['department']['id'] if 'parent' in item else '',
            )

            _process_department(result_departments, departments_to_download, department)

    return result_departments.values()

def export_departments(departments, output):
    rows = ([department_id, parent_department_id, department_name.encode('utf-8')]
            for department_id, department_name, parent_department_id in departments)
    writer = csv.writer(output, delimiter=b';')
    writer.writerow(['orgID', 'ParentID', 'Name'])
    writer.writerows(rows)
