# -*- coding: utf-8 -*-
from __future__ import unicode_literals, absolute_import

import logging
from requests import Session

from competentum import settings

GROUPS_URL = settings.STAFF_API_HOST + '/v3/groups/'
PERSONS_BASE_URL = settings.STAFF_API_HOST + '/v3/persons/'


def download_chiefs_emails(oauth_header, chiefs_logins):
    session = Session()
    session.headers.update(oauth_header)

    def make_params(min_id):
        query_template = 'id > {min_id}'

        return {
            '_sort': 'id',
            '_query': query_template.format(min_id=min_id),
            '_fields': 'id,login,work_email',
            '_limit': 300,
            'login': ','.join(chiefs_logins),
        }

    persons = []
    last_id = -1
    while True:
        params = make_params(last_id)
        logging.info('Getting chief emails with params %s', params)
        response = session.post(url=PERSONS_BASE_URL, data=params).json()
        if not response['result']:
            break
        persons += response['result']
        last_id = response['result'][-1]['id']

    return {person['login']: person['work_email'] for person in persons}


def download_departments_chiefs(oauth_header, url):
    fields = (
        'id',
        'department.id',
        'parent.department.id',
        'department.url',
        'department.heads.role',
        'department.heads.person.login',
    )

    def make_params(min_id):
        query_template = 'id > {min_id} and (department.url == "{url}" or ancestors.department.url == "{url}")'

        return {
            '_sort': 'id',
            '_query': query_template.format(min_id=min_id, url=url),
            '_fields': ','.join(fields),
            '_limit': 300,
            'type': 'department',
        }

    departments = []
    session = Session()
    session.headers.update(oauth_header)

    last_id = -1
    while True:
        params = make_params(last_id)
        logging.info('Downloading departments with params %s', params)
        response = session.get(url=GROUPS_URL, params=params).json()
        if not response['result']:
            break
        departments += response['result']
        last_id = response['result'][-1]['id']

    result = build_id_to_dep(departments)
    emails = download_chiefs_emails(oauth_header, [dep['head'] for dep_id, dep in result.items() if dep['head']])
    for dep_id, dep in result.items():
        if dep['head']:
            dep['head_work_email'] = emails[dep['head']]

    return result


def build_id_to_dep(deps):
    result = {}
    for dep in deps:
        parent_id = dep.get('parent') and dep['parent']['department']['id']
        head = next((role for role in dep['department']['heads'] if role['role'] == 'chief'), None)
        result[dep['department']['id']] = {
            'id': dep['department']['id'],
            'parent_id': parent_id,
            'url': dep['department']['url'],
            'head': head and head['person']['login'],
        }

    return result


def chiefs_chain(id_to_dep, dep_id):
    dep = id_to_dep.get(dep_id)
    if not dep:
        return []

    if not dep['head']:
        return chiefs_chain(id_to_dep, dep['parent_id'])

    return [(dep['head'], dep['head_work_email'])] + chiefs_chain(id_to_dep, dep['parent_id'])


def chief(login, dep_id, id_to_dep):
    for chief_login, chief_work_email in chiefs_chain(id_to_dep, dep_id):
        if chief_login != login:
            return chief_login, chief_work_email
    return None, None
