# -*- coding: utf-8 -*-

from __future__ import unicode_literals, absolute_import

import csv
from logging import getLogger

from requests import get, post, Timeout

from competentum import settings
from competentum.report.departments_chiefs import download_departments_chiefs, chief
from competentum.report.person import Person

logger = getLogger(__name__)

DATE_LEN = len('YYYY-MM-DD')

COUNTRIES = [
    'Россия',
    'Беларусь',
    'Казахстан',
]

BLACK_LIST_OF_ORGANIZATIONS = [
    'Яндекс.Украина',
    'Yandex.Turkey (Reklamcilik)',
    'Yandex.Technology (Germany)',
    'Yandex.Europe (Turkey)',
    'Yandex (Netherlands)',
    'Yandex.Europe (Netherlands)',
    'Yandex.Europe (Switzerland)',
    'Yandex.Zurich',
    'Yandex.Oy (Finland)',
    'Yandex.Services (Switzerland)',
    'Yandex Inc.',
    'Yandex Information Technology (Shanghai)',
    'SPB Software',
    'Yandex Israel Ltd',
    'Yandex.Taxi (Netherlands)',
    'Яндекс.Такси Украина',
    'Яндекс.Израиль',
]

PERSONS_FIELDS = [
    'login',
    'name',
    'work_email',
    'official.position.ru',
    'department_group.department.name.full.ru',
    'department_group.department.id',
    'department_group.parent.department.id',
    'personal.gender',
    'personal.birthday',
    'work_phone',
    'created_at',
    'uid',
    'location.office.name.ru',
    'official.organization.name',
]

PERSONS_BASE_URL = settings.STAFF_API_HOST + '/v3/persons/'


def _get_persons(data, header):
    try:
        logger.info('Geting data for persons %s', data)
        response = post(
            PERSONS_BASE_URL,
            data=data,
            headers=header,
            timeout=60,
            allow_redirects=False,
        )
    except Timeout:
        logger.error(
            'Timeout on connect to the %s',
            PERSONS_BASE_URL,
        )
        raise

    return response.json()


def download_persons(url, header):
    query = (
        "official.organization.name not in {organizations} \
        and (department_group.url == '{url}' or department_group.ancestors.url == '{url}')"
            .format(url=url, organizations=BLACK_LIST_OF_ORGANIZATIONS)
    )

    params = {
        '_fields': ','.join(PERSONS_FIELDS),
        'official.is_robot': False,
        'official.is_dismissed': False,
        'official.affiliation': 'yandex',
        'location.office.city.country.name.ru': ','.join(COUNTRIES),
        '_limit': 100000,
        '_page': 1,
        '_query': query,
    }

    result = _get_persons(params, header)

    for person in result['result']:
        yield person


def _parse_location(office):
    if office == 'Надомник':
        return office

    data = office.split(',', 2)

    if len(data) == 1:
        return '{office}/Неизвестно'.format(office=office)

    if len(data) == 0:
        return ''

    office_name = data[-1].strip()
    city = data[-2]
    return '{city}/{office_name}'.format(city=city, office_name=office_name)


def _raw_to_person(person, departments_heads):
    login = person['login']
    first_name = person['name']['first']['ru']
    last_name = person['name']['last']['ru']
    department_id = person['department_group']['department']['id']
    head_login, head_email = chief(person['login'], department_id, departments_heads)

    work_phone = person['work_phone']
    work_email = person['work_email']
    hire_date = person['created_at'][0:DATE_LEN]
    gender = 1 if person['personal']['gender'] == 'male' else 0
    birthday = person['personal']['birthday']
    position = person['official']['position']['ru']
    location = _parse_location(person['location']['office']['name']['ru'])
    organization = person['official']['organization'].get('name', 'Не указанo')

    try:
        parent_department_id = person['department_group']['parent']['department']['id']
    except:
        parent_department_id = ''

    department_name = person['department_group']['department']['name']['full']['ru']

    return Person(
        login=login,
        last_name=last_name,
        first_name=first_name,
        middle_name='',
        work_email=work_email,
        comment='',
        gender=gender,
        birthday=birthday,
        head_login=head_login,
        head_email=head_email,
        work_phone=work_phone,
        address='',
        additional_info='',
        password='',
        hire_date=hire_date,
        position=position,
        department_id=department_id,
        role='',
        inherit='',
        location=location,
        organization=organization,
        parent_department_id=parent_department_id,
        department_name=department_name,
    )


def _get_persons_in_maternity_vacation(header):
    logger.info('Downloading persons in maternity')
    get_maternity_url = (settings.STAFF_HOST + '/gap-api/api/export_gaps/'
                                               '?workflow=maternity&field=id')

    try:
        response = get(get_maternity_url, headers=header, timeout=5).json()
    except Timeout:
        logger.error(
            'Timeout on connect to the %s',
            get_maternity_url,
        )
        raise

    return {key for key in response['persons']}


def _download_missing_heads(header, result, departments_heads):
    def missing_heads(persons):
        result_logins = {person.login for person in persons}
        result_heads = {person.head_login for person in persons if person.head_login}
        return result_heads.difference(result_logins)

    to_download = missing_heads(result)
    params = {
        '_fields': ','.join(PERSONS_FIELDS),
        'login': ','.join(to_download),
    }
    persons = _get_persons(params, header)
    heads = [_raw_to_person(person, departments_heads) for person in persons['result']]
    return heads


def _filter_out_persons_in_maternity(persons, header):
    all_heads = {person.head_login for person in persons}
    persons_in_maternity = _get_persons_in_maternity_vacation(header)
    return [
        person
        for person in persons
        if person.login not in persons_in_maternity or person.login in all_heads
    ]


def get_persons(url, header):
    department_heads = download_departments_chiefs(header, url)

    all_persons = [
        _raw_to_person(person, department_heads)
        for person in download_persons(url, header)
    ]

    result = _filter_out_persons_in_maternity(all_persons, header)
    result = result + _download_missing_heads(header, result, department_heads)
    result = [person for person in result if person.login not in ('volozh',)]
    return result


def export_persons(persons, output):
    def persons_to_users_row(persons):
        for person in persons:
            assert isinstance(person, Person)
            full_name = '{} {}'.format(person.last_name, person.first_name)

            result = [
                'i:0#.f|aspnetmembership|' + person.work_email,
                person.work_email,
                full_name,
                person.location,
                person.organization,
                person.work_email,
                person.head_email,
                person.comment,
                '',
                '',
                '',
                person.address,
                person.additional_info,
                person.hire_date,
            ]

            yield [
                field.encode('utf-8') if isinstance(field, unicode) else field
                for field in result
            ]

    writer = csv.writer(output, delimiter=b';')
    writer.writerow([
        'User account',
        'ID',
        'Last Name',
        'First Name',
        'Middle Name',
        'Email',
        'Manager ID',
        'Comment',
        'Gender',
        'Birthday',
        'Work Phone',
        'Address',
        'Additional Info',
        'HireDate',
    ])
    writer.writerows(persons_to_users_row(persons=persons))
