from dataclasses import dataclass
import datetime
import logging
from collections import defaultdict
from typing import Any, Dict, List

from aiopg import sa

from src.budget_positions.models import budget_position_fact_table
from src.common import SqlExecutor

logger = logging.getLogger(__name__)


@dataclass
class FilterParams:
    department_id: int or None
    budget_position_code: int or None
    event_time__ge: datetime.datetime or None
    event_time__le: datetime.datetime or None
    login: str or None
    event_types: List[str] or None


@dataclass
class GetBudgetPositionsParams:
    continuation_token: str or None
    limit: int
    filter_params: FilterParams



class DwhStorage:
    def __init__(
        self,
        engine: sa.engine.Engine,
        executor: SqlExecutor = None,
    ) -> None:
        self._engine = engine
        self._executor = executor or SqlExecutor()

    async def get_budget_positions(self, params: GetBudgetPositionsParams) -> Dict[str, Any]:
        sorting = ['-event_time', 'bp_code', 'id']
        where_clause: Dict[str, Dict[str, Any]] = defaultdict(dict)

        if params.filter_params.department_id is not None:
            where_clause['department_id']['eq'] = params.filter_params.department_id

        if params.filter_params.login is not None:
            where_clause['login']['eq'] = params.filter_params.login

        if params.filter_params.budget_position_code is not None:
            where_clause['bp_code']['eq'] = params.filter_params.budget_position_code

        if params.filter_params.event_time__ge is not None:
            where_clause['event_time']['ge'] = params.filter_params.event_time__ge

        if params.filter_params.event_time__le is not None:
            where_clause['event_time']['le'] = params.filter_params.event_time__le

        if params.filter_params.event_types:
            where_clause['event_type']['in'] = params.filter_params.event_types

        logger.info(f'Retrieving budget positions for {where_clause}')
        result = []

        async with self._engine.acquire() as connection:
            facts = self._executor.select(
                connection,
                budget_position_fact_table,
                where_clause,
                sorting,
                params.limit,
                params.continuation_token,
            )

            known_facts = {}
            async for fact in facts:
                bp_code = fact['bp_code']
                result.append({'bp_code': bp_code, 'was': None, 'is': fact, 'debug_info': ''})
                next_fact_position = known_facts.get(bp_code)
                known_facts[bp_code] = len(result) - 1

                if next_fact_position is not None:
                    result[next_fact_position]['was'] = fact

        logger.info(f'{len(result)} budget positions retrieved for {where_clause}')

        return {
            'data': result,
            'continuation_token': self._executor.get_continuation_token(result, sorting, params.limit),
        }
