from dateutil import parser as date_parser

from starlette.datastructures import QueryParams
from starlette.responses import PlainTextResponse
from starlette.requests import Request

from src.budget_positions.dwh_storage import (
    DwhStorage as BudgetPositionsStorage,
    GetBudgetPositionsParams,
    FilterParams,
)
from src.common import JsonResponse
from src.common.access import has_permissions
from src.common.decorators import auth_by_tvm_only


MAX_PAGE_SIZE = 1000


def _parse_get_budget_positions_params(query_params: QueryParams) -> GetBudgetPositionsParams or None:
    department_id = query_params.get('department_id')
    budget_position_code = query_params.get('budget_position_code')
    event_time__ge = query_params.get('event_time__ge')
    event_time__le = query_params.get('event_time__le')
    login = query_params.get('login')
    event_types = query_params.getlist('event_type')

    try:
        if department_id is not None:
            department_id = int(department_id)

        if budget_position_code is not None:
            budget_position_code = int(budget_position_code)

        if event_time__ge is not None:
            event_time__ge = date_parser.parse(event_time__ge)

        if event_time__le is not None:
            event_time__le = date_parser.parse(event_time__le)

        event_types = [str(item) for item in event_types] if event_types else None
        continuation_token = query_params.get('continuation_token')
        limit = query_params.get('limit')

        if limit:
            limit = min(MAX_PAGE_SIZE, int(limit))
            if limit <= 0:
                raise ValueError
        else:
            limit = MAX_PAGE_SIZE

        filter_params = FilterParams(
            department_id=department_id,
            budget_position_code=budget_position_code,
            event_time__ge=event_time__ge,
            event_time__le=event_time__le,
            login=login,
            event_types=event_types,
        )
        return GetBudgetPositionsParams(
            filter_params=filter_params,
            continuation_token=continuation_token,
            limit=limit,
        )
    except (ValueError, date_parser.ParserError):
        return None


@auth_by_tvm_only
async def get_budget_positions(request: Request):
    if not await has_permissions(request.state.user, role='hr_analyst'):
        return PlainTextResponse('hr_analyst only', 403)

    get_budget_positions_params = _parse_get_budget_positions_params(request.query_params)

    if not get_budget_positions_params:
        return PlainTextResponse('Invalid parameters', 400)

    storage = BudgetPositionsStorage(request.app.state.engine)
    result = await storage.get_budget_positions(get_budget_positions_params)
    return JsonResponse(result)

