from typing import Union

import aiohttp
from tvm2 import TVM2Deploy
from src.config import settings
from tvmauth import BlackboxTvmId as BlackboxClientId


async def get_real_ip(headers):
    return headers.get(settings.REAL_IP_HEADER)


async def get_params(uid):
    return {
        "method": "userinfo",
        "format": "json",
        "uid": f"{uid}",
        "userip": '127.0.0.1',
    }


async def get_tvm_client():
    return TVM2Deploy(
        blackbox_client=BlackboxClientId.ProdYateam,
        client_id=settings.YAUTH_TVM2_CLIENT_ID,
    )


async def get_tvm_service_ticket(destination: str):
    params = {'dsts': destination, 'src': settings.YAUTH_TVM2_CLIENT_ID}
    timeout = aiohttp.ClientTimeout(total=settings.GET_TVM_TICKET_TIMEOUT)
    session = aiohttp.ClientSession(headers=settings.LOCAL_TVM_AUTH_HEADERS, timeout=timeout)

    async with session:
        async with session.get(url=settings.TVM_LOCAL_ADDRESS_TICKETS, params=params) as response:
            json_response = await response.json()
            return json_response[destination]['ticket']


async def get_tvm_service_ticket_header(destination: str):
    service_ticket = await get_tvm_service_ticket(destination)
    return {
        settings.TVM_SERVICE_TICKET_HEADER: service_ticket,
    }


async def get_user_info_from_bb(ticket):
    parsed_ticket = (await get_tvm_client()).parse_user_ticket(ticket)
    params = await get_params(uid=parsed_ticket.default_uid)
    timeout = aiohttp.ClientTimeout(total=settings.GET_TVM_TICKET_TIMEOUT)
    session = aiohttp.ClientSession(timeout=timeout)
    headers = await get_tvm_service_ticket_header("blackbox")
    session.headers.update(headers)
    async with session:
        async with session.get(url=settings.BLACKBOX_URL, params=params, headers=headers) as response:
            return await response.json()
