import logging
from dateutil import parser as date_parser
from typing import List, Any, Dict

from sqlalchemy import or_, and_

logger = logging.getLogger(__name__)


class ContinuationTokenException(ValueError):
    pass


class ContinuationTokenManager:
    @staticmethod
    def get_filter(table, continuation_token: str or None, actual_sorting: List[str]):
        token_data = continuation_token.split('~')

        if len(token_data) != len(actual_sorting) + 1:
            logger.info('Invalid token %s length for sorting %s', continuation_token, actual_sorting)
            raise ContinuationTokenException

        sorting = token_data[0].split('.')

        if actual_sorting != sorting:
            logger.info('Invalid sorting %s for token %s, %s expected', actual_sorting, continuation_token, sorting)
            raise ContinuationTokenException

        try:
            # TODO: Build where clause based on sorting and type mapping
            event_time = date_parser.parse(token_data[1])
            bp_code = int(token_data[2])
            current_id = int(token_data[3])

            return or_(
                table.c.event_time < event_time,
                and_(table.c.event_time == event_time, table.c.bp_code > bp_code),
                and_(table.c.event_time == event_time, table.c.bp_code == bp_code, table.c.id > current_id),
            )

        except (ValueError, date_parser.ParserError):
            logger.info('Invalid token %s', continuation_token, exc_info=True)
            raise ContinuationTokenException

    @staticmethod
    def get_continuation_token(result: List[Dict[str, Any]], sorting: List[str], limit: int) -> str or None:
        if len(result) < limit:
            return None

        last_result = result[-1]
        last_result_token = f"~{last_result['is']['event_time']}~{last_result['bp_code']}~{last_result['is']['id']}"

        return '.'.join(sorting) + last_result_token
