import aiohttp
import logging

from starlette.applications import Starlette

from src.change_registry.models import change_record
from src.common import FieldMapper, SqlExecutor
from src.config import settings
from src.tvm import TVM_SERVICE_TICKET_HEADER, get_tvm_ticket

logger = logging.getLogger('change_registry.pull_from_registry')


class PullData:
    def __init__(self, mapper: FieldMapper = None, executor: SqlExecutor = None) -> None:
        super().__init__()
        self._mapper = mapper or FieldMapper()
        self._executor = executor or SqlExecutor()

    async def run(self, app: Starlette) -> bool:
        source_url = f'https://{settings.STAFF_HOST}/budget-position-api/export-change-registry/'

        async with app.state.engine.acquire() as connection:
            last_date = await self._executor.get_last_processed_source_date(connection, change_record)
            logger.info(f'Last stable source_date: {last_date}')

            params = {}
            if last_date:
                # TODO: Move to filters
                params['continuation_token'] = last_date.isoformat()

            while True:
                tvm_token = await get_tvm_ticket('staff')
                async with aiohttp.ClientSession(headers={TVM_SERVICE_TICKET_HEADER: tvm_token}) as session:
                    async with session.get(url=source_url, params=params) as response:
                        # TODO: raise for status
                        response = await response.json()
                        source_changes = response['data']

                if not source_changes:
                    logger.info('No data from the source')
                    return True

                logger.info(f'{len(source_changes)} changes from the source')
                for change in source_changes:
                    mapped = self._mapper.map(
                        'change-registry',
                        change,
                        change_record,
                        custom={
                            self._executor.source_date_field_name: 'workflow_confirmed_at',
                            'started_at': 'workflow_created_at',
                            'resolved_at': 'workflow_confirmed_at',
                        },
                    )

                    await self._executor.insert(connection, change_record, mapped)

                continuation_token = response.get('continuation_token', None)
                if continuation_token is None:
                    return False
                else:
                    params['continuation_token'] = continuation_token
