import asyncio
import logging
import time

from starlette.applications import Starlette

from ylog.context import log_context

from src.common import Container, LockProvider, LockNotAvailable, PgAdvisoryLockProvider
from src.change_registry.pull_from_registry import PullData
from src.pull_from_logbroker import PullLogBrokerData


logger = logging.getLogger(__name__)


async def periodic(
    container: Container,
    app: Starlette,
    task_type,
    lock_provider: LockProvider,
    delay: float,
    execution_timeout: int,
):
    task_name = f'{task_type.__module__}.{task_type.__qualname__}'

    while True:
        task = container.resolve(task_type)
        work_finished = True

        with log_context(task_name=task_name, task_id=id(task)):
            try:
                start_time = time.time()
                async with lock_provider.acquire_lock(app, task_name, execution_timeout):
                    logger.info(f'Task {task_name} started')
                    start_time = time.time()

                    work_finished = await task.run(app)
                    logger.info(f'Task {task_name} success in {time.time() - start_time} seconds')
            except LockNotAvailable:
                logger.debug(f'Task {task_name} skipped')
            except Exception as e:
                logger.exception(f'Task {task_name} failed in {time.time() - start_time} seconds: {e}')
            if work_finished:
                await asyncio.sleep(delay)


async def start_tasks(app: Starlette):
    tasks = [(PullData, 1 * 60, 30 * 60), (PullLogBrokerData, 1 * 60, 30 * 60)]

    async def _multiple_tasks():
        return await asyncio.gather(*app.state.tasks, return_exceptions=True)

    container = Container()
    lock_provider = PgAdvisoryLockProvider()
    logger.info('STARTUP: Gathering tasks...')
    app.state.tasks = [periodic(container, app, task[0], lock_provider, task[1], task[2]) for task in tasks]
    asyncio.get_event_loop().create_task(_multiple_tasks())
    logger.info(f'STARTUP: {len(app.state.tasks)} tasks scheduled')


async def shutdown_tasks(app: Starlette):
    logger.info(f'SHUTDOWN: Cancelling tasks...')
    for task in app.state.tasks:
        task.cancel()
    logger.info(f'SHUTDOWN: Tasks cancelled')
