#!/usr/bin/env python
# coding: utf-8
import os
import sys
from logging import getLogger, INFO, basicConfig
import asyncio
import asyncpg
import ssl

logger = getLogger(__name__)
logger.setLevel(INFO)
basicConfig(stream=sys.stdout, level=INFO)


class UdpServer:
    def connection_made(self, transport):
        pass

    def datagram_received(self, data, addr):
        message = data.decode()
        loop.create_task(put_to_database(message))


async def put_to_database(message):
    try:
        # check nginx.conf for log format
        syslog_header, host, login = message.split('|', maxsplit=2)
        if login == '-':
            return

        async with pool.acquire() as connection:
            await connection.execute(
                'INSERT INTO logs(host, login) VALUES ($1, $2)',
                host,
                login.lower(),
            )

    except Exception as e:
        logger.warning(e)


async def create_table():
    async with pool.acquire() as connection:
        logger.info('Creating table')
        await connection.execute(
            'CREATE TABLE IF NOT EXISTS logs ('
                  'timestamp timestamp default current_timestamp,'
                  'host inet,'
                  'login varchar(50))'
        )
        logger.info('Done')

PROD_SETTINGS = {
    'host': [
        'c-ff16e93e-6ae2-4efc-9dce-80ed157c4880.rw.db.yandex.net',
    ],
    'database': 'staff_whistlah',
    'max_size': 21,
}

TEST_SETTINGS = {
    'host': [
        'c-bb2d588f-48d2-4cae-97ca-6f71cadd4725.rw.db.yandex.net',
    ],
    'database': 'staff_whistlah_test',
    'max_size': 29,
}


def create_pool():
    context = ssl.SSLContext(ssl.PROTOCOL_TLSv1_2)
    context.load_verify_locations('/certs/allCAs.pem')

    production = os.environ.get('STAFF_ENV', 'testing') == 'production'
    settings = PROD_SETTINGS if production else TEST_SETTINGS

    return asyncpg.create_pool(
        password=os.environ['WHISTLAH_POSTGRES_PASS'],
        port=6432,
        user='staff_whistlah',
        ssl=context,
        **settings,
    )


if __name__ == '__main__':
    loop = asyncio.get_event_loop()

    logger.info('Starting log server, waiting 5s for database')
    loop.run_until_complete(asyncio.sleep(5))

    logger.info('Connecting to DB')

    pool = loop.run_until_complete(create_pool())

    logger.info('Checking for log table')
    loop.run_until_complete(create_table())

    logger.info('Starting up server')
    listen = loop.create_datagram_endpoint(UdpServer, local_addr=('::', 13113))
    transport, protocol = loop.run_until_complete(listen)

    logger.info('Up and running')
    try:
        loop.run_forever()
    except KeyboardInterrupt:
        pass

    logger.info('Stopping...')
    transport.close()
    loop.close()
