#! -*- coding: utf-8 -*-
import json
import urllib3
import requests
from urllib3.exceptions import InsecureRequestWarning

urllib3.disable_warnings(category=InsecureRequestWarning)


from constants import ONE_REQUEST_PAGE_LIMIT, STAFF_API_URL


class Staff(object):
    def __init__(self, oauth_token):
        self._session = requests.Session()
        self._api_url = '{STAFF_API_URL}/v3/'.format(STAFF_API_URL=STAFF_API_URL)
        self._session.headers.update({
            'Authorization': 'OAuth {}'.format(oauth_token),
        })

    def get_user_info(self, login, many=False):
        """ Returns staff json result with full user info 'as-is' """
        staff_request_params = {
            'login': login,
            'department_group.type': 'department',
            # раньше были указаны в GET_USER_INFO_FIELDS
            '_fields': ','.join([
                'login',
                'uid',
                'name',
                'work_email',
                'language',
                'official.is_dismissed',
                'official.position',
                'official.join_at',
                'department_group.name',
                'chief.login',
                'chiefs.login',
                'hr_partners.login',
            ]),
            '_limit': ONE_REQUEST_PAGE_LIMIT,
            '_one': 1,  # to get entire user info
        }

        if many:
            staff_request_params.pop('_one')
        else:
            staff_request_params.pop('_fields')

        staff_response = self._session.post(self._api_url + "person", params=staff_request_params, verify=False)

        if staff_response.status_code != 200:
            return {'status_code': staff_response.status_code}

        return staff_response.json()

    def get_user_groups(self, login):
        """ Returns groups where user is in ( including ancestor ones ) """
        staff_request_params = {
            'login': login,
            'official.is_dismissed': False,
            'department_group.type': 'department',
            '_fields': ','.join([
                'id',
                'login',
                'groups.group.id',
                'groups.group.name',
                'groups.group.type',
                'groups.group.is_deleted',
                'groups.group.level',
                'groups.group.ancestors.id',
                'groups.group.ancestors.name',
                'groups.group.ancestors.type',
                'groups.group.ancestors.is_deleted',
                'groups.group.ancestors.level',
            ]),
            '_limit': ONE_REQUEST_PAGE_LIMIT,
        }
        staff_response = self._session.post(self._api_url + "person", params=staff_request_params, verify=False)

        if staff_response.status_code != 200:
            return {'status_code': staff_response.status_code}

        return json.dumps(staff_response.json().get('result', []))

    def get_group_users_recursive(self, group_url):
        """
        Returns group users ( including users in nested groups ) -
        makes several recursive calls for nested deps structure
        """
        groups_to_search = [group_url]

        def _get_nested_groups(group_url):
            staff_request_params = {
                'parent.url': group_url
            }
            staff_response = (
                self._session.post(self._api_url + "groups", params=staff_request_params, verify=False).json()
            )
            nested_result = staff_response.get('result', [])
            for nested_group in nested_result:
                nested_group_url = nested_group['url']
                groups_to_search.append(nested_group_url)
                _get_nested_groups(nested_group_url)
        _get_nested_groups(group_url)

        results = []

        for group in groups_to_search:
            staff_request_params = {
                'department_group.url': group,
                'department_group.type': 'department',
                'official.is_dismissed': False
            }
            staff_response = self._session.post(self._api_url + "persons", params=staff_request_params, verify=False)

            if staff_response.status_code != 200:
                return {'status_code': staff_response.status_code}

            results += staff_response.json()['result']
        return json.dumps(results)

    def _get_group_users(self, group_url):
        """ Returns group users ( including users in nested groups ) """
        staff_request_params = [
            {
                '_fields': 'login',
                '_sort': 'id',
                '_limit': ONE_REQUEST_PAGE_LIMIT,  # to avoid paging

                'is_deleted': False,
                'department_group.ancestors.department.url': group_url,
                'official.is_dismissed': False
            },
            {
                '_fields': 'login',
                '_sort': 'id',
                '_limit': ONE_REQUEST_PAGE_LIMIT,  # to avoid paging

                'is_deleted': False,
                'department_group.department.url': group_url,
                'official.is_dismissed': False
            }
        ]

        merged_result = []

        for params in staff_request_params:
            staff_response = self._session.post(self._api_url + "persons", params=params, verify=False)

            if staff_response.status_code != 200:
                return {'status_code': staff_response.status_code}

            merged_result += staff_response.json().get('result', [])

        return merged_result

    def get_group_users(self, group_url):
        return json.dumps(self._get_group_users(group_url))

    def get_user_location(self, login):
        """ Returns office with city of user """
        staff_request_params = {
            'login': login,
            'official.is_dismissed': False,
            '_fields': 'location',
            '_limit': ONE_REQUEST_PAGE_LIMIT,
        }
        staff_response = self._session.post(self._api_url + "person", params=staff_request_params, verify=False)

        if staff_response.status_code != 200:
            return {'status_code': staff_response.status_code}

        return json.dumps(staff_response.json().get('result', []))

    def get_hrbp_supervised_persons(self, login):
        """ Returns supervised users for given hrbp login """
        """ DEPRECATED METHOD """

        staff_request_params = [
            {
                'official.is_dismissed': False,
                'department_group.ancestors.department.heads.person.login': login,
                'is_deleted': False,

                '_limit': ONE_REQUEST_PAGE_LIMIT,  # to avoid paging
                '_sort': 'id',  # to avoid cache missing due to random order results
                '_fields': 'login',
            },
            {
                'official.is_dismissed': False,
                'department_group.department.heads.person.login': login,
                'is_deleted': False,

                '_limit': ONE_REQUEST_PAGE_LIMIT,  # to avoid paging
                '_sort': 'id',  # to avoid cache missing due to random order results
                '_fields': 'login',
            }
        ]

        merged_result = []

        for params in staff_request_params:
            staff_response = self._session.post(self._api_url + "person", params=params, verify=False)
            if staff_response.status_code != 200:
                return {'status_code': staff_response.status_code}

            merged_result += staff_response.json().get('result', [])

        return json.dumps(merged_result)

    def get_supervised_persons(self, login, role='chief'):
        """ Returns supervised users for given role """

        staff_request_params = {
            '_query': 'department.heads==match({{"person.login":"{login}","role":"{role}"}})'.format(
                login=login,
                role=role,
            ),
            '_fields': 'department.url',
            '_limit': 100 * 1000,
        }

        staff_response = self._session.get(self._api_url + "groups", params=staff_request_params, verify=False)
        if staff_response.status_code != 200:
            return {'status_code': staff_response.status_code}

        groups = staff_response.json().get('result', [])

        results = []

        for group in groups:
            results += self._get_group_users(group_url=group['department']['url'])

        return json.dumps(results)

    def get_user_roles(self, login):
        """ Get dict with user roles """

        roles = {
            'autoheads': 'chief',
            'hr_partners': 'hrbp',
        }

        result = {
            'chief': False,
            'hr_partner': False,
        }

        auto_groups = self.get_auto_groups_membership(login, need_json=True)['result']

        for group in auto_groups:
            if not roles:
                break
            group_url = group['group'].get('url', '')
            if group_url in roles.keys():
                result[
                    roles[group_url]
                ] = True
                del roles[group_url]

        return json.dumps(result)

    def get_groups(self, page, limit):
        """Returns groups"""
        """ Returns supervised users for given role """

        staff_request_params = {
            '_fields': 'name,id,is_deleted,type,parent.name,level,parent.level,url',
            '_sort': 'id',
            'type': 'department',
        }
        if limit:
            staff_request_params['_limit'] = limit
        if page:
            staff_request_params['_page'] = page

        staff_response = self._session.post(self._api_url + "groups", params=staff_request_params, verify=False)
        if staff_response.status_code != 200:
            return {'status_code': staff_response.status_code}

        response = staff_response.json()

        del response['links']

        return json.dumps(response)

    def get_offices(self, page, limit):
        """Returns offices"""

        staff_request_params = {
            '_fields': 'id,name,is_deleted,city,code,contacts,is_virtual,address,timezone,floors',
            '_sort': 'id',
        }
        if limit:
            staff_request_params['_limit'] = limit
        if page:
            staff_request_params['_page'] = page

        staff_response = self._session.post(self._api_url + "offices", params=staff_request_params, verify=False)
        if staff_response.status_code != 200:
            return {'status_code': staff_response.status_code}

        response = staff_response.json()

        del response['links']

        return json.dumps(response)

    def get_auto_groups_membership(self, login, need_json=False):
        """
        Проксирует вызов staffapi ручки 'groupmembership' для заданного логина.
        Может быть использовано, например, для получения информации о том, когда
        пользователь стал руководителем или HR-партнером.
        """
        staff_request_params = {
            'person.login': login,
            'group.type': 'wiki',
            'group.is_deleted': False,
            '_sort': 'group.id',
            '_fields': 'group.url,group.id,joined_at,group.name',
            '_limit': 1000,
        }
        staff_response = self._session.post(
            self._api_url + "groupmembership",
            params=staff_request_params,
            verify=False
        )
        if staff_response.status_code != 200:
            return {'status_code': staff_response.status_code}

        response = staff_response.json()

        if need_json:
            return response

        return json.dumps(response)

    def has_department_role(self, login, role):
        staff_request_params = {
            'person.login': login,
            'role': role,
            '_limit': 1,
            '_fields': 'id',
        }
        staff_response = self._session.post(
            self._api_url + "departmentstaff",
            params=staff_request_params,
            verify=False
        )
        if staff_response.status_code != 200:
            return {'status_code': staff_response.status_code}

        response = staff_response.json()

        return json.dumps({'has_department_role': response['total'] != 0})


    def get_chief_users(self, login, with_nested=False):
        staff_request_params = {
            'official.is_dismissed': False,
            '_limit': ONE_REQUEST_PAGE_LIMIT,
            '{}.login'.format('chiefs' if with_nested else 'chief'): login,
            '_fields': 'login',
        }

        staff_response = self._session.get(
            self._api_url + "persons",
            params=staff_request_params,
            verify=False
        )

        if staff_response.status_code != 200:
            return {'status_code': staff_response.status_code}

        response = staff_response.json()

        return json.dumps(response['result'])

    def get_hr_partner_users(self, login):
        staff_request_params = {
            'official.is_dismissed': False,
            '_limit': ONE_REQUEST_PAGE_LIMIT,
            'hr_partners.login': login,
            '_fields': 'login',
        }

        staff_response = self._session.get(
            self._api_url + "persons",
            params=staff_request_params,
            verify=False
        )
        if staff_response.status_code != 200:
            return {'status_code': staff_response.status_code}

        response = staff_response.json()

        return json.dumps(response['result'])
