import base64

from django import forms
from django.contrib import admin
from django.core.urlresolvers import reverse

from staff.lib.forms.fields import StaffField
from staff.django_intranet_notifications.utils.wiki import WikiFormatter

from staff.lib.utils.admin import StaffModelAdmin

from . import models


wf = WikiFormatter()


class IconLink(object):
    def __init__(self, instance, size):
        self.url = reverse('achievery:icon_raw',
                           kwargs={'pk': instance.pk, 'size': size})
        self.data = getattr(instance, '%s_data' % size)

    def __str__(self):
        return self.url


class IconAdminForm(forms.ModelForm):
    big_data = forms.FileField(required=True)
    small_data = forms.FileField(required=True)

    def __init__(self, *args, **kwargs):
        instance = kwargs.get('instance')

        initial = {}

        if instance and instance.big_data:
            initial['big_data'] = IconLink(instance, 'big')

        if instance and instance.small_data:
            initial['small_data'] = IconLink(instance, 'small')

        if initial:
            kwargs['initial'] = initial

        super(IconAdminForm, self).__init__(*args, **kwargs)

    def _clean_icon(self, icon):
        if isinstance(icon, IconLink):
            return icon.data
        else:
            return base64.b64encode(icon.read())

    def clean(self):
        cd = self.cleaned_data
        for size in ('big', 'small'):
            attr = '%s_data' % size
            if attr in cd:
                cd[attr] = self._clean_icon(cd[attr])
        return cd

    class Meta:
        model = models.Icon
        fields = '__all__'


class IconAdmin(StaffModelAdmin):
    list_display = (
        '__str__',
        'achievement',
        'level',
    )
    form = IconAdminForm


class IconInline(admin.TabularInline):
    model = models.Icon
    form = IconAdminForm


class AchievementAdminForm(forms.ModelForm):
    is_active = forms.BooleanField(
        initial=False,
        required=False,
        widget=forms.CheckboxInput({'disabled': 'disabled'}))


class AchievementAdmin(StaffModelAdmin):
    group_field = 'owner_group'
    list_display = (
        'i_title',
        'category',
        'owner_group__url',
        'i_service_name',
    )
    list_filter = (
        'category',
        'is_active',
    )

    inlines = (IconInline,)

    def get_form(self, request, obj=None, **kwargs):
        if not request.user.has_perm('achievery.change_achievement_status'):
            kwargs['form'] = AchievementAdminForm
        return super(AchievementAdmin, self).get_form(request, obj, **kwargs)

    def i_title(self, obj):
        return obj.i_title

    def i_service_name(self, obj):
        return obj.i_service_name

    def owner_group__url(self, obj):
        return obj.owner_group.url

    def save_model(self, request, obj, form, change):
        if not request.user.has_perm('achievery.change_achievement_status'):
            obj.is_active = models.Achievement.objects.get(pk=obj.pk).is_active if obj.pk else False
        if obj.description.strip():
            obj.description_html = wf.to_html(obj.description)
        if obj.description_en.strip():
            obj.description_html_en = wf.to_html(obj.description_en)

        super(AchievementAdmin, self).save_model(request, obj, form, change)


class EventMixin(object):
    staff_field = 'initiator'


class EventAdmin(StaffModelAdmin):
    staff_field = 'initiator'

    list_display = (
        'given_achievement',
        'initiator',
        'created_at',
    )
    readonly_fields = ('given_achievement', )


class EventInline(admin.TabularInline, EventMixin):
    model = models.Event

    def formfield_for_foreignkey(self, db_field, request=None, **kwargs):
        if db_field.name == self.staff_field:
            return StaffField(label=db_field.verbose_name)
        return super(EventInline, self).formfield_for_foreignkey(db_field, request, **kwargs)


class GivenAchievementAdmin(StaffModelAdmin):
    staff_field = 'person'
    list_display = (
        'achievement',
        'person',
        'slot',
        'level',
        'is_active',
        'is_hidden',
        'created_at',
        'modified_at',
    )
    list_filter = (
        'is_active',
        'is_hidden',
    )

    inlines = (EventInline,)


admin.site.register(models.Achievement, AchievementAdmin)
admin.site.register(models.GivenAchievement, GivenAchievementAdmin)
admin.site.register(models.Icon, IconAdmin)
admin.site.register(models.Event, EventAdmin)
