import base64
import io
import requests
from typing import Tuple

from PIL import Image

from django.conf import settings
from django.core.exceptions import ValidationError

from staff.achievery.models import Icon


class IconCtl:

    ICON_SIZE_BIG = 100
    ICON_SIZE_SMALL = 50

    def __init__(self, obj: Icon):
        self.obj = obj

    @classmethod
    def get_icon_data_from_url(cls, url: str) -> Tuple[bytes, bytes]:
        errors = []

        image = cls._get_image_by_url(url)

        if min(image.width, image.height) < cls.ICON_SIZE_BIG:
            errors.append(
                ValidationError(f'Icon size should be greater of equal {cls.ICON_SIZE_BIG}px x {cls.ICON_SIZE_BIG}px'),
            )

        # соотношение сторон не более чем 4:3
        if min(image.width / image.height, image.height / image.width) < 0.75:
            errors.append(ValidationError('Bad icon aspect ratio, should be from 1:1 to 4:3 or 3:4'))

        if image.format != 'PNG':
            errors.append(ValidationError('Icon format must be .PNG'))

        if errors:
            raise ValidationError(errors)

        small_data = cls._get_image_data(cls._get_resized_image(image, cls.ICON_SIZE_SMALL))
        big_data = cls._get_image_data(cls._get_resized_image(image, cls.ICON_SIZE_BIG))

        return small_data, big_data

    @staticmethod
    def _get_image_by_url(icon_url) -> Image:
        headers = {
            'Authorization': f'OAuth {settings.ROBOT_STAFF_OAUTH_TOKEN}',
        }
        icon_raw = requests.get(url=icon_url, headers=headers).content
        return Image.open(io.BytesIO(icon_raw))

    @staticmethod
    def _get_resized_image(image: Image, size: int) -> Image:
        image_copy = image.copy()
        image_copy.thumbnail((size, size), Image.LANCZOS)
        return image_copy

    @staticmethod
    def _get_image_data(image: Image) -> bytes:
        img_byte_arr = io.BytesIO()
        image.save(img_byte_arr, format='PNG')
        return base64.b64encode(img_byte_arr.getvalue())
