from staff.user_settings.domain import get_user_settings

from .. import models, schema, exceptions
from ..fields import (
    PlainField,
    RelatedField,
    RelatedListField,
    SlotField,
    LevelField,
    IconField,
    WikiField,
)
from ..notifications import Notifier

from .base import (
    SavableDomainObject,
    CreatableDomainObjectList,
    TimeStamped,
    SoftDeleted,
)
from .event import Event
from .registry import domain_objects, domain_object_lists


__all__ = (
    'GivenAchievement', 'GivenAchievementList',
)


@domain_objects.register
class GivenAchievement(TimeStamped, SoftDeleted, SavableDomainObject):
    model_class = 'achievery.GivenAchievement'

    __diff_exclude_fields__ = ('revision', 'is_hidden', 'slot')

    comment = WikiField('comment_html', schema_type=schema.TYPE.STRING)
    comment_html = PlainField(schema_type=schema.TYPE.STRING)
    is_hidden = PlainField(schema_type=schema.TYPE.BOOLEAN)
    level = LevelField(schema_type=schema.TYPE.NUMBER)
    revision = PlainField(schema_type=schema.TYPE.NUMBER, null=True)
    slot = SlotField(schema_type=schema.TYPE.NUMBER, null=True)
    achievement = RelatedField(related_class='Achievement', assignable=True)
    person = RelatedField(related_class='Person', assignable=True)
    events = RelatedListField(
        related_class='Event', model_field='event_set', sorting=('-id',),
    )
    icon_big = IconField(is_big=True, null=True)
    icon_small = IconField(is_big=False, null=True)
    last_event_at = PlainField(null=True, schema_type=schema.TYPE.STRING)

    def validate_level(self):
        levels = self.model.achievement.icon_set.values_list('level', flat=True)
        if self.level not in levels:
            raise exceptions.UnacceptableLevel(self.level, data=self.level)

    def get_diff_fields(self):
        return set(self.__fields__) - set(self.__diff_exclude_fields__)

    def pre_save(self, force_insert=False, force_update=False):
        super(GivenAchievement, self).pre_save(force_insert, force_update)
        self.revision += 1

        if self.id is None:
            # noinspection PyUnresolvedReferences
            user_settings = get_user_settings(self.person.model)
            self.is_hidden = user_settings.hide_achievements

        if self.slot is not None and (not self.is_active or self.is_hidden):
            self.slot = None

        elif (self.slot is None and self.is_active and not self.is_hidden and self.id is None):
            try:
                # noinspection PyUnresolvedReferences
                free_slot = self.person.free_slots[0]
            except IndexError:
                free_slot = None
            self.slot = free_slot

    def maybe_create_event(self):
        # noinspection PyProtectedMember
        loggable_fields = Event.get_model_class().__data_fields__
        should_create_event = False
        if self.events:
            # noinspection PyUnresolvedReferences
            previous_event = self.events[0]
            diff_fields = self.get_diff_fields()

            changed_fields = [
                f for f in loggable_fields
                if getattr(self.model, f) != getattr(previous_event, f)
                and f in diff_fields
            ]

            if changed_fields:
                should_create_event = True
        else:
            should_create_event = True
        if should_create_event:
            model = models.Event.objects.create(
                given_achievement=self.model,
                initiator=self.user,
                **{f: getattr(self.model, f) for f in loggable_fields}
            )

            self.model.last_event_at = model.created_at
            self.model.save()

            return Event(self.user, model, self.role_registry)

    def maybe_send_notification(self, event):
        if event is not None:
            Notifier(self).try_to_send_something(event)

    def post_save(self, force_insert=False, force_update=False):
        event = self.maybe_create_event()
        self.maybe_send_notification(event)

    def delete(self):
        self.soft_delete()
        self.save()
        self.role_registry.fill()

    def restore(self):
        self.soft_restore()
        self.save()
        self.role_registry.fill()


@domain_object_lists.register
class GivenAchievementList(CreatableDomainObjectList):
    domain_object_class = GivenAchievement
