from staff.lib.utils.ordered_choices import OrderedChoices
from staff.lib.utils.reflection import classproperty
from staff.lib.utils.text import camel_to_underscore


SCOPE = OrderedChoices(
    ('PARAM', 'parameters'),
    ('FIELD', 'fields'),
    ('GLOBAL', 'global'),
)


class AchieveryError(Exception):
    status = 500
    text = 'achievery.error.unknown'
    scope = SCOPE.GLOBAL

    def __init__(self, *args, **kwargs):
        self.data = kwargs.pop('data', None)
        self.cause = kwargs.pop('cause', None)
        super(AchieveryError, self).__init__(*args, **kwargs)

    # noinspection PyMethodParameters
    @classproperty
    def code(cls):
        return camel_to_underscore(cls.__name__)

    @property
    def text(self):
        return (
            'achievery.error.%s' % camel_to_underscore(self.__class__.__name__)
        )

    def __str__(self):
        return ' '.join(filter(bool, (
            super(AchieveryError, self).__str__(),
            str(self.data) if self.data else '',
            ('caused by %s%s' % (self.cause.__class__.__name__, self.cause)
             if self.cause
             else '')
        )))


class RequestError(AchieveryError):
    status = 400


class PermissionError(AchieveryError):
    status = 403


class DoesNotExist(AchieveryError):
    status = 404


class BusinessError(AchieveryError):
    status = 409


class ValidationError(BusinessError):
    """Domain object validation failed"""
    scope = SCOPE.FIELD


class DuplicateEntry(BusinessError):
    """Object parameter conflicts with existing"""


class ColumnCannotBeNull(BusinessError):
    """Database error: column cannot be NULL"""


class ConcurrentModification(BusinessError):
    """Revision of object does not match in request and database"""


class LookupTooAbstract(RequestError):
    """Lookup is too abstract"""
    scope = SCOPE.PARAM
    text = 'achievery.error.lookup_too_abstract'


class FieldNotFound(RequestError):
    """Requested field does not exist"""
    scope = SCOPE.PARAM
    text = 'achievery.error.field_not_found'


class RequiredFieldIsMissing(ValidationError):
    """Domain object field is not null, but null is passed"""


class UnacceptableLevel(ValidationError):
    """GivenAchievement with that level cannot be created"""


class InconsistentIcons(ValidationError):
    """All icons must go in pairs: big + small"""
    scope = SCOPE.GLOBAL


class InvalidSlot(ValidationError):
    """Slot value is illegal"""
