from operator import attrgetter

from django.core.exceptions import ObjectDoesNotExist

from .. import schema

from .plain import PlainField


class RelatedField(PlainField):
    DEFAULT_SCHEMA_TYPE = schema.TYPE.OBJECT

    def __init__(self, *args, **kwargs):
        self.related_class = kwargs.pop('related_class', None)
        self.pseudo = kwargs.pop('pseudo', False)
        self.assignable = kwargs.pop('assignable', False)
        super(RelatedField, self).__init__(*args, **kwargs)

    def get_related_class(self):
        from ..domain import domain_objects
        return domain_objects[self.related_class]

    def get_related_model(self, instance, owner):
        if not self.pseudo:
            return super(RelatedField, self).get(instance, owner)
        else:
            return instance.model

    def get(self, instance, owner):
        try:
            model = self.get_related_model(instance, owner)
        except ObjectDoesNotExist:
            return None
        else:
            related_class = self.get_related_class()
            return related_class(instance.user, model, instance.role_registry)

    def set(self, instance, value):
        if not self.assignable:
            raise AttributeError('%s.%s does not support assignment' %
                                 (instance.__class__.__name__, self.name))

        if not isinstance(value, self.get_related_class()):
            raise TypeError('%s must be of type %s, but it is %s' %
                            (self.name, self.get_related_class(), type(value)))

        super(RelatedField, self).set(instance, value.model)

    def translate_lookup(self, attrs, value, source_field):
        related_class = self.get_related_class()
        if attrs:
            attr = attrs[0]
            nested = attrs[1:]
            k, v = related_class.__dict__[attr].translate_lookup(
                nested, value, source_field)

            if self.pseudo:
                return k, v

            return (
                '%s__%s' % (self.name, k),
                v
            )
        elif isinstance(value, related_class):
            return (
                '%s__id' % self.name,
                value.id
            )

    @property
    def schema(self):
        return self.get_related_class().schema


class RelatedListField(PlainField):
    DEFAULT_SCHEMA_TYPE = schema.TYPE.ARRAY

    def __init__(self, *args, **kwargs):
        self.model_field = kwargs.pop('model_field', None)
        self.related_class = kwargs.pop('related_class', None)
        self.lookup = kwargs.pop('lookup', {})
        self.single = kwargs.pop('single', False)
        self.sorting = kwargs.pop('sorting', ())
        super(RelatedListField, self).__init__(*args, **kwargs)

    def get_related_class(self):
        from ..domain import domain_object_lists
        return domain_object_lists[self.related_class]

    def get(self, instance, owner):
        getter = attrgetter(self.model_field)

        try:
            manager = getter(instance.model)
        except ObjectDoesNotExist:
            return None if self.single else ()

        lookup = self.resolve_lookup(self.lookup, instance)
        qs = manager.filter(**lookup).order_by(*self.sorting)
        results = self.get_related_class()(
            user=instance.user,
            queryset=qs,
            role_registry=instance.role_registry,
        )

        if self.single:
            return results[0] if results else None
        else:
            return results

    def set(self, instance, value):
        raise AttributeError(self.name)

    def translate_lookup(self, attrs, value, source_field):
        attr = attrs[0]
        nested = attrs[1:]
        related = self.related_class.__dict__[attr]
        k, v = related.translate_lookup(nested, value, source_field)
        return (
            '%s__%s' % (self.model_field, k),
            v
        )

    @property
    def schema(self):
        return {
            'type': self.schema_type,
            'items': self.get_related_class().domain_object_class.schema,
        }

    def resolve_lookup(self, lookup, instance):
        if callable(lookup):
            return lookup(instance)
        return lookup
