import warnings

from staff.lib.utils.reflection import classproperty

from .. import schema
from ..exceptions import LookupTooAbstract


class ComplexFieldAdapter(object):
    def __init__(self, model, spec):
        self.model = model
        self.spec = spec

    def __getattr__(self, item):
        if item == '__iter__':  # Why have I do this?..
            raise AttributeError()

        if item in self.spec:
            return MultilingualAdapter(self.model, self.spec[item])

    def __dir__(self):
        return list(self.spec.keys())

    @classmethod
    def translate_lookup(cls, attrs, value, spec, source_field):
        field = attrs[0]
        nested = attrs[1:]
        return MultilingualAdapter.translate_lookup(
            nested, value, spec[field], source_field)

    @classmethod
    def fields(cls, spec):
        return {
            field: MultilingualAdapter.fields
            for field in spec
        }


class MultilingualAdapter(object):
    def __init__(self, model, field_name):
        self.__dict__['model'] = model
        self.__dict__['name'] = field_name

    def __getattr__(self, item):
        if item == 'en':
            return getattr(self.model, self.name + '_en')

        elif item == self.model.native_lang:
            return getattr(self.model, self.name)

        else:
            if not self.model.native_lang:
                warnings.warn(
                    '`native_lang` field of %s is empty, '
                    'but it should not be' % self.model
                )
            raise AttributeError(item)

    def __setattr__(self, key, value):
        if key == 'en':
            setattr(self.model, self.name + '_en', value)

        elif key == self.model.native_lang:
            setattr(self.model, self.name, value)

        else:
            raise AttributeError(key, value)

    def __dir__(self):
        return list(self.iterdir())

    def iterdir(self):
        if hasattr(self.model, self.name + '_en'):
            yield 'en'

        if hasattr(self.model, self.name) and self.model.native_lang:
            yield self.model.native_lang

    def items(self):
        for k in self.iterdir():
            yield k, getattr(self, k)

    def __str__(self):
        return '{}'.format(self.en)

    @classmethod
    def translate_lookup(cls, attrs, value, field_name, source_field):
        try:
            attr = attrs[0]
        except IndexError:
            raise LookupTooAbstract(source_field, data=source_field)

        if attr == 'en':
            return field_name + '_en', value
        else:
            return field_name, value

    # noinspection PyMethodParameters
    @classproperty
    def fields(cls):
        return {  # TODO: what if native_lang == 'tr'?
            'ru': {'type': schema.TYPE.STRING},
            'en': {'type': schema.TYPE.STRING},
        }

    def update(self, data):
        for field_name in self.fields:
            try:
                value = data[field_name]
            except KeyError:
                continue
            else:
                setattr(self, field_name, value)
