from django.core.exceptions import ObjectDoesNotExist
from django.db import models

from staff.lib.models.base import I18nModel

from staff.lib.models.base import TimeStampedModel, SoftDeletedModel
from staff.lib.models.managers import FilterManager
from staff.lib.utils.library import Library

from .constants import CATEGORY, MIME_TYPE


TEXT_LENGTH_TINY = 16
TEXT_LENGTH_SMALL = 255
TEXT_LENGTH_NORMAL = 1024


library = Library(lambda x: x.__name__)


class Queries:
    def __init__(self, queries=None, **kwargs):
        self._queries = queries or kwargs or {}

    def __getattr__(self, item):
        if item == 'contribute_to_class':
            raise AttributeError
        return self._queries[item]


@library.register
class Achievement(I18nModel, TimeStampedModel, SoftDeletedModel):

    category = models.CharField(max_length=TEXT_LENGTH_TINY, choices=CATEGORY)
    description = models.TextField(blank=True)
    description_html = models.TextField(blank=True)
    description_en = models.TextField(blank=True)
    description_html_en = models.TextField(blank=True)

    description_short = models.CharField(
        max_length=TEXT_LENGTH_SMALL,
        blank=True,
    )

    description_short_en = models.CharField(
        max_length=TEXT_LENGTH_SMALL,
        blank=True,
    )

    # noinspection PyUnresolvedReferences
    owner_group = models.ForeignKey('django_intranet_stuff.Group')
    service_name = models.CharField(max_length=TEXT_LENGTH_SMALL, blank=True)
    service_name_en = models.CharField(
        max_length=TEXT_LENGTH_SMALL,
        blank=True,
    )

    title = models.CharField(max_length=TEXT_LENGTH_SMALL, unique=True)
    title_en = models.CharField(max_length=TEXT_LENGTH_SMALL, unique=True)

    queries = Queries(
        active={'is_active': True},
        inactive={'is_active': False},
        owned_by=lambda user: {'owner_group__groupmembership__staff': user}
    )

    def __str__(self):
        return "{}".format(self.i_title)

    class Meta:
        permissions = (
            ('change_achievement_status', 'Can change "is_active" field'),
        )


@library.register
class GivenAchievement(TimeStampedModel, SoftDeletedModel):
    achievement = models.ForeignKey('Achievement')

    # noinspection PyUnresolvedReferences
    person = models.ForeignKey('django_intranet_stuff.Staff')
    comment = models.TextField(blank=True)
    comment_html = models.TextField(blank=True)
    is_hidden = models.BooleanField(default=False)
    level = models.IntegerField()
    revision = models.IntegerField(default=0)
    slot = models.IntegerField(null=True, blank=True)
    last_event_at = models.DateTimeField(null=True)

    queries = Queries(
        hidden={'is_hidden': True, 'is_active': True},
        active={'is_hidden': False, 'is_active': True},
        inactive={'is_active': False},
        held_by=lambda user: {'person': user},
        owned_by=lambda user: {
            'achievement__owner_group__groupmembership__staff': user,
        },
    )

    objects = models.Manager()
    hidden = FilterManager(**queries.hidden)
    active = FilterManager(**queries.active)
    inactive = FilterManager(**queries.inactive)

    def __str__(self):
        # TODO: some utility encapsulating that ugly try/except ObjectDoesNotExist
        try:
            achievement = str(self.achievement)
        except ObjectDoesNotExist:
            achievement = '?'

        try:
            person = str(self.person)
        except ObjectDoesNotExist:
            person = '?'

        return '{} | {}'.format(achievement, person)

    class Meta:
        unique_together = (
            ('person', 'slot'),
            ('person', 'achievement'),
        )


@library.register
class Icon(models.Model):
    big_data = models.TextField(default='')
    small_data = models.TextField(default='')
    mime_type = models.CharField(
        max_length=TEXT_LENGTH_TINY,
        choices=MIME_TYPE,
        blank=True,
        default=MIME_TYPE.PNG_BASE64,
    )
    level = models.IntegerField()
    achievement = models.ForeignKey('Achievement')
    modified_at = models.DateTimeField(auto_now=True)

    queries = Queries(
        default_big={'level': -1},
        default_small={'level': -1},

        related_big=lambda instance: {'level': instance.level},
        related_small=lambda instance: {'level': instance.level},
    )

    def __str__(self):
        return '{} | {} '.format(self.achievement, self.level)


@library.register
class Event(TimeStampedModel):
    __data_fields__ = (
        'comment', 'is_active', 'is_hidden', 'level', 'revision', 'slot',
    )

    comment = models.TextField()
    given_achievement = models.ForeignKey('GivenAchievement')
    # noinspection PyUnresolvedReferences
    initiator = models.ForeignKey('django_intranet_stuff.Staff')
    is_active = models.BooleanField(default=False)
    is_hidden = models.BooleanField(default=False)
    level = models.IntegerField()
    revision = models.IntegerField()

    slot = models.IntegerField(
        null=True,
        blank=True,
    )

    def __str__(self):
        try:
            given_achievement = str(self.given_achievement)
        except ObjectDoesNotExist:
            given_achievement = '?'

        try:
            initiator = str(self.initiator)
        except ObjectDoesNotExist:
            initiator = '?'

        return '[{}] by {} at {}'.format(
            given_achievement, initiator, self.created_at
        )


class Permissions(models.Model):
    class Meta:
        managed = False
        default_permissions = ()  # disable "add", "change", "delete" default permissions

        permissions = (
            ('external_with_achievery_access', 'External with achievery access'),
        )
