import logging
import yenv

from django.conf import settings
from django.core.urlresolvers import reverse

from staff.django_intranet_notifications import Notification
from staff.lib.utils.text import camel_to_underscore
from staff.lib.utils.library import Library
from staff.person.models import Staff

from staff.achievery.models import Achievement


library = Library(lambda n: n.get_target())
logger = logging.getLogger(__name__)


def maybe_delay(task, *args, **kwargs):
    if yenv.type == 'development':
        return task(*args, **kwargs)
    else:
        return task.apply_async(args=args, kwargs=kwargs, countdown=20)


class Notifier(object):
    def __init__(self, given):
        self.given = given

    def try_to_send_something(self, event):
        for notification in filter(bool, (n(event) for n in library)):
            AsyncSender(notification).send_async()
            break  # Одного письма хватит для любых целей


class AsyncSender(object):
    def __init__(self, notification):
        self.notification = notification

    def serializable_params(self):
        return {
            'event_id': self.notification.event.id,
            'target': self.notification.get_target()
        }

    def send_async(self):
        from .tasks import celery_send_mail
        params = self.serializable_params()
        maybe_delay(celery_send_mail, **params)


class NotificationContext(object):
    protocol = settings.STAFF_PROTOCOL
    host = settings.STAFF_HOST

    def __init__(self, event, lang):
        self.event = event
        self.lang = lang

    def as_dict(self):
        def resolve(value):
            return value() if callable(value) else value

        return {
            k: resolve(getattr(self, k))
            for k in dir(self)
            if not k.startswith('_') and k != 'as_dict'
        }

    @property
    def _staff_url(self):
        return '{protocol}://{host}'.format(
            protocol=self.protocol, host=self.host)

    def description_en(self):
        return self.event.given.achievement.description.en

    def description_ru(self):
        return self.event.given.achievement.description.ru

    def description_short_en(self):
        return self.event.given.achievement.description_short.en

    def description_short_ru(self):
        return self.event.given.achievement.description_short.ru

    def title_en(self):
        return self.event.given.achievement.title.en

    def title_ru(self):
        return self.event.given.achievement.title.ru

    def title(self):
        return getattr(self.event.given.achievement.title, self.lang)

    def given_url(self):
        return '{base}/achievements/users/{login}#{id}'.format(
            base=self._staff_url, login=self.event.given.person.login,
            id=self.event.given.achievement.id,
        )

    def icon_url(self):
        url = '{base}{icon}'.format(
            base=self._staff_url,
            icon=reverse(
                'achievery:icon_raw',
                kwargs={
                    'pk': self.event.given.icon_big.id,
                    'size': 'big',
                }
            )
        )
        return url

    def url(self):
        return '{base}/achievements/achievement/{id}'.format(
            base=self._staff_url, id=self.event.given.achievement.id)

    def comment(self):
        return self.event.given.comment

    def comment_html(self):
        return self.event.given.comment_html

    def level(self):
        level = self.event.given.level
        return None if level == -1 else level

    def initiator_url(self):
        return '{base}/{login}'.format(
            base=self._staff_url, login=self.event.initiator.login
        )

    def initiator_name_en(self):
        return '{name.first.en} {name.last.en}'.format(
            name=self.event.initiator.name)

    def initiator_name_ru(self):
        return '{name.first.ru} {name.last.ru}'.format(
            name=self.event.initiator.name)

    def initiator_name(self):
        return '{} {}'.format(
            getattr(self.event.initiator.name.first, self.lang),
            getattr(self.event.initiator.name.last, self.lang),
        )

    def initiator_name_ablative(self):
        from django.utils import translation
        with translation.override(self.lang):
            return self.event.initiator.inflections.ablative

    def is_hidden(self):
        return self.event.is_hidden


class DINNotificationAdapter(Notification):
    def __init__(self, *args, **kwargs):
        self.template = kwargs.pop('template', None)
        logger.info(self.template)
        self.id = kwargs.pop('id', None)
        super(DINNotificationAdapter, self).__init__(*args, **kwargs)

    def get_subj_id(self):
        return 'achievery_%s_%s' % (self.target, self.id)


class AchieveryNotification(object):
    def __init__(self, event):
        self.event = event

    def get_lang(self):
        return self.event.given.person.lang_ui

    def get_template(self):
        return 'achievery/{lang}/{name}.html'.format(
            name=camel_to_underscore(self.__class__.__name__),
            lang=self.get_lang()
        )

    @classmethod
    def get_target(cls):
        return camel_to_underscore(cls.__name__).upper()

    def __bool__(self):
        return self.should_be_sent()

    def should_be_sent(self):
        raise NotImplementedError('Subclass must implement this')

    def send(self):
        din_notification = DINNotificationAdapter(
            target=self.get_target(),
            context=self.get_context(),
            template=self.get_template(),
            id=self.event.id,
        )
        din_notification.send(recipients=self.get_recipients())

    def get_context(self):
        return NotificationContext(self.event, lang=self.get_lang()).as_dict()

    def get_recipients(self):
        return [self.event.given.person.work_email]


class AchievementCreated(Notification):

    def __init__(self, achievement: Achievement, creator: Staff, *args, **kwargs):
        kwargs['target'] = 'achievement_created'
        kwargs['context'] = {
            'achievement': achievement,
            'staff_host': settings.STAFF_HOST,
            'idm_host': settings.IDM_HOST,
        }
        kwargs['template'] = f'achievery/{creator.lang_ui}/{kwargs["target"]}.html'

        super().__init__(*args, **kwargs)

    def get_subj_id(self) -> str:
        return f'achievement_created_{self.context["achievement"].id}'


class AchievementCreationFailed(Notification):

    def __init__(self, creator: Staff, *args, **kwargs):
        kwargs['target'] = 'achievement_creation_failed'
        self.creator_login = creator.login
        kwargs['template'] = f'achievery/{creator.lang_ui}/{kwargs["target"]}.html'

        super().__init__(*args, **kwargs)

    def get_subj_id(self) -> str:
        return f'achievement_creation_failed_{self.creator_login}'


@library.register
class AchievementTookAway(AchieveryNotification):
    def should_be_sent(self):
        return self.event.is_taken_away


@library.register
class AchievementUnlocked(AchieveryNotification):
    def should_be_sent(self):
        return self.event.is_unlocked


@library.register
class Levelup(AchieveryNotification):
    def should_be_sent(self):
        return self.event.is_levelup


@library.register
class Leveldown(AchieveryNotification):
    def should_be_sent(self):
        return self.event.is_leveldown


@library.register
class AchievementReturned(AchieveryNotification):
    def should_be_sent(self):
        return self.event.is_returned
