from itertools import chain, product
import logging

from staff.lib.models import smart_model

from .roles.base import Condition

from . import models, roles


logger = logging.getLogger(__name__)


class PermissionChecker(object):
    def __init__(self, roles_list):
        self.roles = list(roles_list)

    def is_field_readable(self, field):
        return any(r.is_field_readable(field) for r in self.roles)

    def is_field_writable(self, field):
        return any(r.is_field_readable(field) for r in self.roles)

    def check(self, field, perm):
        return any(r.check(field, perm) for r in self.roles)

    def __str__(self):
        return '{}({})'.format(self.__class__.__name__, self.roles)


def model_is_new(model):
    return model.id is None or getattr(model, '_is_new_', False)


class RoleRegistry(object):
    def __init__(self, user):
        self._cache = {}
        self.user = user

        self._role_model = set()
        if self.user:
            self.fill()

        self.user_owner_for = None

    def is_user_owner_for(self, achievement_id):
        if self.user_owner_for is None:
            self.user_owner_for = set(
                models.Achievement.objects
                .filter(owner_group__groupmembership__staff=self.user)
                .values_list('id', flat=True)
            )

        return achievement_id in self.user_owner_for

    def potential_roles(self, user, model):
        if user == self.user:
            if model_is_new(model):
                return []
            else:
                return (
                    Role(user, model)
                    for Role, Model in self._role_model
                    if isinstance(model, Model) and Role.check_object(self, model)
                )
        else:
            return (r(user, model) for r in roles.library)

    def get_bool_map(self, model):
        return dict(chain(
            [(roles.Admin.name, bool(roles.Admin(self.user, model.__class__)))],
            (
                (
                    Role.name,
                    not model_is_new(model) and Role.check_object(self, model)
                )
                for Role in roles.library
            )
        ))

    def get(self, user, model):
        if user == self.user:
            role_classes = []

            if model.__class__ not in models.library:
                role_classes.append(roles.Unmanaged)

            if roles.Creation(user, model).is_applicable():
                role_classes.append(roles.Creation)

            if not model_is_new(model):
                role_classes.extend(
                    Role for Role, Model in self._role_model
                    if isinstance(model, Model) and Role.check_object(self, model)
                )

            return PermissionChecker(r(user, model) for r in role_classes)

        else:
            key = (user.id, model.__class__.__name__, model.id)
            if key not in self._cache:
                self._cache[key] = PermissionChecker(
                    list(filter(bool, self.potential_roles(user, model)))
                )
            return self._cache[key]

    def __getitem__(self, item):
        user, model = item
        return self.get(user, model)

    @classmethod
    def get_base_query(cls, user, model_class):
        role_classes = cls.roles_maybe_with_admin(user)
        if model_class not in models.library:
            role_classes.append(roles.Unmanaged)
        return sum(
            (r.get_query(user, smart_model(model_class)) for r in role_classes),
            Condition(none=True)
        )

    @classmethod
    def roles_maybe_with_admin(cls, user, model_class=None):
        if model_class and model_class not in models.library:
            return [roles.Unmanaged]
        roles_list = list(roles.library)
        if roles.Admin(user, model_class).is_applicable():
            roles_list.append(roles.Admin)
        return roles_list

    def fill(self):
        roles_list = self.roles_maybe_with_admin(self.user)
        for Role, Model in product(roles_list, models.library):
            if not Role.manages(Model):
                continue
            self._role_model.add((Role, Model))
