from staff.achievery import models

from .base import Condition, Role, ANONYMOUS
from .registry import library


class Holder(Role):
    """Абстрактный носитель"""

    # TODO: Объединить логику методов `is_applicable` и `get_query`
    # Она неким образом дублируется
    def is_applicable(self):
        if not isinstance(self.model, models.GivenAchievement):
            return False

        return self.model.person == self.user

    @classmethod
    def get_query(cls, user, model_class):
        return {
            models.GivenAchievement:
                Condition(models.GivenAchievement.queries.held_by(user)),
        }.get(model_class, Condition(none=True))

    @classmethod
    def check_object(cls, role_registry, obj):
        if not isinstance(obj, models.GivenAchievement):
            return False
        return obj.person_id == role_registry.user.id


@library.register
class ActiveHolder(Holder):
    """Носитель активной ачивки"""

    fields = {
        models.GivenAchievement: {
            'is_hidden': 'w',
            'slot': 'w',
            'revision': 'w'
        },
    }

    def is_applicable(self):
        return (super(ActiveHolder, self).is_applicable()
                and self.model.is_active
                and not self.model.is_hidden)

    @classmethod
    def get_query(cls, user, model_class):
        if user is ANONYMOUS:
            return Condition(none=True)

        holder_cond = super(ActiveHolder, cls).get_query(user, model_class)
        return holder_cond & Condition(models.GivenAchievement.queries.active)

    @classmethod
    def check_object(cls, role_registry, obj):
        if role_registry.user is ANONYMOUS:
            return False

        return (
            super(ActiveHolder, cls).check_object(role_registry, obj)
            and obj.is_active
        )


@library.register
class HiddenHolder(Holder):
    """Носитель скрытой ачивки"""

    fields = {
        models.GivenAchievement: {
            'id': 'r',
            'comment': 'r',
            'comment_html': 'r',
            'is_hidden': 'rw',
            'is_active': 'r',
            'level': 'r',
            'revision': 'rw',
            'slot': 'r',
            'achievement': 'r',
            'person': 'r',
            'events': 'r',
            'created_at': 'r',
            'modified_at': 'r',
            'last_event_at': 'r',
        },
    }

    def is_applicable(self):
        return (super(HiddenHolder, self).is_applicable()
                and self.model.is_active
                and self.model.is_hidden)

    @classmethod
    def get_query(cls, user, model_class):
        if user is ANONYMOUS:
            return Condition(none=True)

        holder_cond = super(HiddenHolder, cls).get_query(user, model_class)
        return holder_cond & Condition(models.GivenAchievement.queries.hidden)

    @classmethod
    def check_object(cls, role_registry, obj):
        if role_registry.user is ANONYMOUS:
            return False

        return (
            super(HiddenHolder, cls).check_object(role_registry, obj)
            and obj.is_hidden
        )


@library.register
class InactiveHolder(Holder):
    """Носитель неактивной ачивки"""

    fields = {
        models.GivenAchievement: {
            'id': 'r',
            'comment': 'r',
            'comment_html': 'r',
            'is_hidden': 'r',
            'is_active': 'r',
            'level': 'r',
            'revision': 'r',
            'slot': 'r',
            'achievement': 'r',
            'person': 'r',
            'events': 'r',
            'created_at': 'r',
            'modified_at': 'r',
            'last_event_at': 'r',
        },
    }

    def is_applicable(self):
        return (super(InactiveHolder, self).is_applicable()
                and not self.model.is_active)

    @classmethod
    def get_query(cls, user, model_class):
        if user is ANONYMOUS:
            return Condition(none=True)

        holder_cond = super(InactiveHolder, cls).get_query(user, model_class)
        return holder_cond & Condition(models.GivenAchievement.queries.inactive)

    @classmethod
    def check_object(cls, role_registry, obj):
        if role_registry.user is ANONYMOUS:
            return False

        return (
            super(InactiveHolder, cls).check_object(role_registry, obj)
            and not obj.is_active
        )
