import logging

from rest_framework import serializers

from .base import DynamicFieldsSerializer


logger = logging.getLogger(__name__)


class LangPairSerializer(DynamicFieldsSerializer):
    ru = serializers.CharField(required=False)
    en = serializers.CharField(required=False)
    localized = serializers.SerializerMethodField('_get_localized')

    def to_native(self, obj):
        # Patch self.fields to avoid AttributeError
        # when serializing an object with native_lang = ''
        allowed = set(dir(obj))
        if self.context.get('lang') is not None:
            allowed.add('localized')

        existing = set(self.fields.keys())
        backup = self.fields.copy()
        for field in existing - allowed:
            logger.debug('Disabling field %s', field)
            self.fields.pop(field)

        try:
            result = super(LangPairSerializer, self).to_native(obj)
        finally:
            self.fields = backup

        return result

    def _get_localized(self, obj):
        requested_lang = self.context.get('lang')
        return getattr(obj, requested_lang, getattr(obj, 'en'))
