import logging

from django.core.exceptions import ValidationError

from staff.celery_app import app
from staff.groups.models import Group
from staff.groups.objects.controller import WikiGroupCtl
from staff.person.models import Staff

from staff.achievery import notifications, permissions
from staff.achievery.controllers.icon import IconCtl
from staff.achievery.domain import Event
from staff.achievery.forms.create_achievement import CreateAchievementForm
from staff.achievery.models import Achievement, Icon
from staff.achievery.notifications import AchievementCreated, AchievementCreationFailed
from staff.achievery.utils import flatten

logger = logging.getLogger(__name__)


@app.task
def celery_send_mail(event_id, target):
    model = Event.get_model_class().objects.get(id=event_id)
    roles = permissions.RoleRegistry(model.initiator)
    event = Event(model.initiator, model, roles)
    notification = notifications.library[target](event)
    notification.send()


@app.task(ignore_result=True)
def create_achievement_task(data: dict):
    form = CreateAchievementForm(icon_required=True, data=data)

    errors = []

    if not form.is_valid():
        errors.extend(list(flatten(form.errors.values())))
    else:
        try:
            small_icon_data, big_icon_data = IconCtl.get_icon_data_from_url(form.cleaned_data['icon_url'])
        except ValidationError as err:
            errors.extend(err.messages)

    if errors:
        try:
            creator = Staff.objects.get(login=data['creator_login'])
            AchievementCreationFailed(creator, context={'errors': errors}).send(recipients=[creator.work_email])
        except (KeyError, Staff.DoesNotExist):
            logger.warning(
                'Can not create achievement because of wrong or empty login %s',
                data.get('login', ''),
            )
    else:
        owner_group = WikiGroupCtl().create(parent=Group.objects.get(url='__wiki__'), commit=True)

        achievement = Achievement.objects.create(owner_group=owner_group, **form.achievement_data)
        Icon.objects.create(
            achievement=achievement,
            level=-1,
            big_data=big_icon_data,
            small_data=small_icon_data,
        )

        WikiGroupCtl(owner_group).update(
            {
                'name': f'achievery_{achievement.id}',
                'url': f'achievery_{achievement.id}',
                'code': f'achievery_{achievement.id}',
                'description': f'Group for {achievement.title} achievement management',
            },
        )

        AchievementCreated(achievement, form.cleaned_data['creator']).send(
            recipients=[form.cleaned_data['creator'].work_email],
        )
        logger.info('Successfully created achievement %s', achievement.id)
