from collections import defaultdict
from email.utils import formatdate
import time

from django.conf import settings

from staff.person.models import Staff

from staff.achievery.models import GivenAchievement


class ChainedDefaultDict(defaultdict):
    def __init__(self, default_factory=None, **kwargs):
        default_factory = default_factory or ChainedDefaultDict
        super(ChainedDefaultDict, self).__init__(default_factory, **kwargs)

    def __getitem__(self, key):
        if '.' not in key:
            return super(ChainedDefaultDict, self).__getitem__(key)
        else:
            first, other = key.split('.', 1)
            return self[first][other]

    def __setitem__(self, key, value):
        if '.' not in key:
            super(ChainedDefaultDict, self).__setitem__(key, value)
        else:
            first, other = key.split('.', 1)
            self[first][other] = value

    def to_dict(self):
        return {
            k: v.to_dict() if hasattr(v, 'to_dict') else v
            for k, v in self.items()
        }


def expand(flat):
    expanded = ChainedDefaultDict()
    for key, val in sorted(flat.items(), reverse=True):
        expanded[key.replace('__', '.')] = val
    return expanded.to_dict()


def subfields(fields, root):
    if not fields:
        return fields
    root += '.'
    length = len(root)
    return [f[length:] for f in fields if f.startswith(root)]


def flatten_dict(dict_, prefix=''):
    for k, v in dict_.items():
        if hasattr(v, 'items'):
            # yield from
            for subitem in flatten_dict(v, '.'.join((prefix, k))):
                yield subitem
        else:
            yield '.'.join(filter(bool, [prefix, k, v]))


def list_paths(struct, prefix=''):
    def cat(*bits): return '.'.join(filter(bool, bits))

    if isinstance(struct, str):
        yield cat(prefix, struct)

    elif hasattr(struct, 'items'):
        for k, v in struct.items():
            yield cat(prefix, k)
            for item in list_paths(v, cat(prefix, k)):
                yield item

    elif hasattr(struct, '__iter__'):
        for v in struct:
            for item in list_paths(v, prefix):
                yield item


def flatten(seq):
    for item in seq:
        if isinstance(item, (tuple, list)):
            # yield from
            for subitem in flatten(item):
                yield subitem
        else:
            yield item


class IconModelWrapper(object):
    __model_fields__ = (
        'id', 'mime_type', 'level', 'achievement', 'modified_at',
    )

    def __init__(self, model, is_big):
        self._model = model
        self._is_big = is_big

    def __getattr__(self, item):
        if item in self.__model_fields__:
            return getattr(self._model, item)
        raise AttributeError(item)

    @property
    def data(self):
        attr = '%s_data' % ('big' if self._is_big else 'small')
        return getattr(self._model, attr)

    @property
    def is_big(self):
        return self._is_big


def icon_compat_wrapper(icon_qs):
    for icon in icon_qs:
        base = {
            k: icon[k] for k in ('mime_type', 'level', 'achievement_id', 'id')
        }
        small = base.copy()
        small.update(data=icon.get('small_data', ''), is_big=False)
        yield small

        big = base.copy()
        big.update(data=icon.get('big_data', ''), is_big=True)
        yield big


def datetime2rfc(dt):
    dt = time.mktime(dt.timetuple())
    return formatdate(dt, usegmt=True)


def find_available_slot(person: Staff):
    occupied_slots = (
        GivenAchievement.objects
        .filter(person=person, slot__isnull=False)
        .filter(slot__gte=1)
        .values_list('slot', flat=True)
    )
    for i in range(1, settings.ACHIEVERY_MAX_SLOTS + 1):
        if i not in occupied_slots:
            return i
    return 0
