from itertools import product

from django.core.paginator import Paginator, EmptyPage
from django.db.models import Q, Count, Value

from rest_framework.response import Response

from staff.groups.models import Group, GroupMembership

from staff.lib.utils.qs_values import extract_related

from staff.achievery.domain import Achievement
from staff.achievery.models import Icon, GivenAchievement
from staff.achievery.serializers import PaginatedAchievementSerializer, AchievementSerializer
from staff.achievery.utils import icon_compat_wrapper, expand
from staff.achievery.views.base import (
    AchieveryAPIView,
    ListMixin,
    SchemaView,
    DetailsMixin,
    EditMixin,
)
from staff.achievery.views.utils import serialize_achievement, ACHIEVEMENT_FIELDS


__all__ = (
    'AchievementListView', 'AchievementSchemaView', 'AchievementDetailsView'
)


class AchievementView(AchieveryAPIView):
    domain_object_class = Achievement


class AchievementListView(AchievementView, ListMixin):
    paginated_serializer_class = PaginatedAchievementSerializer
    default_fields = ACHIEVEMENT_FIELDS

    def get(self, request):
        lookup = self.validate_fields(request.PARAMS.filter)
        user = self.get_user()
        roles = self.role_registry
        model_cls = Achievement.get_model_class()
        filter_lookup = list(Achievement.translate_lookup(lookup))
        sort = list(Achievement.translate_fields(request.PARAMS.sort))

        achievements = (
            roles
            .get_base_query(user, model_cls)
            .get_queryset(model_cls)
            .filter(*filter_lookup)
            .order_by(*sort)
        )
        if request.PARAMS.search:
            search_lookup = Q()
            for f, suffix in product(Achievement.__search_fields__, Achievement.__search_suffixes__):
                search_lookup |= Q(**{'{}__{}'.format(f, suffix): request.PARAMS.search})
            achievements = achievements.filter(search_lookup)

        paginator = Paginator(achievements.values().distinct(),
                              request.PARAMS.limit)
        try:
            page = paginator.page(request.PARAMS.page)
        except EmptyPage:
            page = paginator.page(paginator.num_pages)

        achievements = page.object_list

        achievements_ids = set()
        groups_ids = set()
        for achievement in achievements:
            achievements_ids.add(achievement['id'])
            groups_ids.add(achievement['owner_group_id'])

        group_memberships = (
            GroupMembership.objects
            .filter(group__in=groups_ids)
            .values(
                'group_id',
                'staff__id',
                'staff__login',
                'staff__uid',
                'staff__is_dismissed',
                'staff__first_name',
                'staff__first_name_en',
                'staff__last_name',
                'staff__last_name_en'
            )
        )

        persons_by_groups = {
            gm['group_id']: extract_related(gm, 'staff')
            for gm in group_memberships
        }

        given_counts = {
            ga['achievement']: ga['count'] for ga in
            GivenAchievement.active.values('achievement').annotate(count=Count(Value(1)))
        }

        groups = Group.objects.filter(id__in=groups_ids)

        ctx = {
            'groups': {g['id']: g for g in groups.values()},
            'icons': {
                (i['achievement_id'], i['level'], i['is_big']): i for i
                in icon_compat_wrapper(
                    Icon.objects
                    .filter(achievement__in=achievements_ids)
                    .values(
                        'id',
                        'mime_type',
                        'level',
                        'achievement_id',
                        'modified_at'
                    )
                )
            },
            'persons_by_groups': persons_by_groups,
            'localize': self.get_lang(),
            'role_registry': self.role_registry,
            'given_counts': given_counts,
        }

        requested_fields = expand(dict.fromkeys(request.PARAMS.fields)) or self.default_fields
        data = {
            'total': paginator.count,
            'limit': paginator.per_page,
            'pages': paginator.num_pages,
            'page': page.number,
            'objects': [
                serialize_achievement(obj, ctx, fields=requested_fields)
                for obj in achievements
            ],
        }

        return Response(data)


class AchievementDetailsView(AchievementView, DetailsMixin, EditMixin):
    serializer_class = AchievementSerializer


class AchievementSchemaView(SchemaView):
    domain_object_class = Achievement
