from django.core.paginator import Paginator, EmptyPage

from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response
from rest_framework.views import APIView

from ..exceptions import AchieveryError, FieldNotFound
from ..permissions import RoleRegistry
from ..request import AchieveryAPIRequest as Request
from ..roles.base import ANONYMOUS
from ..serializers import ExceptionBlockSerializer
from ..utils import expand

import logging


logger = logging.getLogger(__name__)


class AchieveryAPIView(APIView):
    permission_classes = (IsAuthenticated,)

    def initialize_request(self, request, *args, **kwargs):
        parser_context = self.get_parser_context(request)
        request._dont_enforce_csrf_checks = True

        return Request(request,
                       parsers=self.get_parsers(),
                       authenticators=self.get_authenticators(),
                       negotiator=self.get_content_negotiator(),
                       parser_context=parser_context)

    def handle_exception(self, exc):
        if isinstance(exc, AchieveryError):
            if exc.status >= 500:
                logger.exception('AchieveryError %s %s: %s', exc.status, exc.text, exc)
            else:
                logger.info('AchieveryError %s %s: %s', exc.status, exc.text, exc, exc_info=True)

            serializer = ExceptionBlockSerializer(exc)
            return Response(serializer.data, status=exc.status)
        else:
            return super(AchieveryAPIView, self).handle_exception(exc)


class SchemaView(AchieveryAPIView):
    domain_object_class = None

    def get(self, request):
        schema = self.domain_object_class.schema
        return Response(schema)


class DomainObjectMixin(object):
    domain_object_class = None
    serializer_class = None
    paginated_serializer_class = None
    _role_registry = None
    _user = None
    request = None

    def validate_fields(self, lookup):
        requested = set(expand(lookup))
        present = set(self.domain_object_class.__fields__)

        unknown = requested - present
        if unknown:
            raise FieldNotFound(data=unknown.pop())

        return lookup

    def get_user(self):
        if self._user is None:
            if self.request.user.is_anonymous():
                self._user = ANONYMOUS
            else:
                self._user = self.request.user.get_profile()
        return self._user

    @property
    def role_registry(self):
        if self._role_registry is None:
            self._role_registry = RoleRegistry(user=self.get_user())
        return self._role_registry

    def get_context(self):
        return {
            'user': self.get_user(),
            'role_registry': self.role_registry,
            'fields': self.get_fields(),
            'lang': self.get_lang(),
        }

    def get_fields(self):
        return self.request.PARAMS.fields

    def get_lang(self):
        if self.request.PARAMS.localize:
            return getattr(self.get_user(), 'lang_ui', 'ru')


class ListMixin(DomainObjectMixin):
    def get(self, request):
        lookup = self.validate_fields(request.PARAMS.filter)
        objects = (
            self.domain_object_class
            .objects(self.get_user(), self.role_registry)
            .filter(**lookup)
            .search(request.PARAMS.search)
            .order_by(*request.PARAMS.sort)
            .distinct()
        )

        paginator = Paginator(objects, request.PARAMS.limit)
        try:
            page = paginator.page(request.PARAMS.page)
        except EmptyPage:
            page = paginator.page(paginator.num_pages)

        serializer = self.paginated_serializer_class(
            page, context=self.get_context()
        )
        return Response(serializer.data)


class DetailsMixin(DomainObjectMixin):
    def get_response(self, data):
        return Response(data)

    def get_object(self, **kwargs):
        instance = (
            self.domain_object_class
            .objects(self.get_user(), self.role_registry)
            .get(pk=kwargs['pk'])
        )
        return instance

    def get(self, request, **kwargs):
        instance = self.get_object(**kwargs)
        serializer = self.serializer_class(
            instance, context=self.get_context()
        )
        return self.get_response(serializer.data)


class EditMixin(DomainObjectMixin):
    def put(self, request, pk):
        instance = self.domain_object_class.objects(
            self.get_user(), self.role_registry
        ).get(pk=pk)
        serializer = self.serializer_class(
            instance,
            data=expand(request.DATA),
            partial=True,
            context=self.get_context(),
        )

        if serializer.is_valid():
            serializer.save()
            self.role_registry.fill()
            return Response(serializer.data)
        return Response(serializer.errors, status=400)

    def post(self, request, pk):
        return self.put(request, pk)


class CreateMixin(DomainObjectMixin):
    def post(self, request):
        instance = self.domain_object_class(
            user=self.get_user(),
            model=self.domain_object_class.get_model_class()(),
            role_registry=self.role_registry,
        )
        serializer = self.serializer_class(
            instance,
            data=expand(request.DATA),
            context=self.get_context(),
        )
        if serializer.is_valid():
            serializer.save()
            self.role_registry.fill()
            return Response(serializer.data, status=201)
        # TODO: wrap serializer errors to render consistently with other errors
        return Response(serializer.errors, status=400)
