from django.shortcuts import get_object_or_404

from rest_framework.permissions import AllowAny
from rest_framework.response import Response

from staff.lib.decorators import available_for_external

from ..domain import Icon
from ..models import GivenAchievement, Icon as IconModel
from ..renderers import PNGRenderer
from ..serializers import IconSerializer

from .base import (
    AchieveryAPIView,
    DomainObjectMixin,
)
from ..utils import (
    IconModelWrapper,
    datetime2rfc,
)


def modified_at_rfc(modified_at):
    return datetime2rfc(modified_at)


@available_for_external('achievery.external_with_achievery_access')
class IconRawView(AchieveryAPIView, DomainObjectMixin):
    permission_classes = (AllowAny,)

    domain_object_class = Icon
    serializer_class = IconSerializer
    renderer_classes = (PNGRenderer,)

    def get_response(self, instance, modified_at):
        serializer = self.serializer_class(instance, context=self.get_context())
        return Response(
            serializer.data,
            content_type='image/png',
            headers={'Last-Modified': modified_at_rfc(modified_at)},
        )

    def get(self, request, pk, size, **kwargs):
        instance = (
            getattr(self.domain_object_class, size)
            (self.get_user(), self.role_registry)
            .get(pk=pk)
        )
        return self.get_response(instance, instance.model.modified_at)


class IconByLoginRawView(IconRawView):
    def get(self, request, yandex_login, achievement_id, size, **kwargs):
        given = get_object_or_404(
            GivenAchievement,
            person__login=yandex_login,
            achievement_id=achievement_id,
        )
        icon = get_object_or_404(
            IconModel,
            achievement_id=achievement_id,
            level=given.level,
        )
        wrapped = IconModelWrapper(icon, is_big=(size == 'big'))
        instance = self.domain_object_class(self.get_user(), wrapped, self.role_registry)

        return self.get_response(instance, given.modified_at)
