from django.contrib import admin
from django.contrib.admin.models import LogEntry, ADDITION, CHANGE, DELETION


class LogEntryAdmin(admin.ModelAdmin):
    list_display = ('action_time', 'user', 'get_action', 'get_url', 'change_message')
    fields = list_display
    readonly_fields = list_display
    list_filter = ('content_type__model',)
    search_fields = ('user__username',)

    def get_action(self, obj):
        actions = {
            ADDITION: 'Добавлено',
            CHANGE: 'Изменено',
            DELETION: 'Удалено',
        }
        return actions.get(obj.action_flag)
    get_action.short_description = 'Действие'  # type: ignore

    def get_url(self, obj):
        return '<a href="{url}">{name}</a>'.format(
            url=obj.get_admin_url(),
            name='{}: {}'.format(obj.content_type.model, obj.object_id),
        )
    get_url.short_description = 'Объект'  # type: ignore
    get_url.allow_tags = True  # type: ignore

    def has_delete_permission(self, request, obj=None):
        return False

    def has_add_permission(self, request, obj=None):
        return False

    def get_actions(self, request):
        actions = super(LogEntryAdmin, self).get_actions(request)
        del actions['delete_selected']
        return actions

    def get_search_results(self, request, queryset, search_term):
        words = search_term.split(' ')
        object_ids = [x for x in words if x.isdigit()]
        usernames = list(set(words) - set(object_ids))

        qs, use_distinct = super(LogEntryAdmin, self).get_search_results(request, queryset, ' '.join(usernames))

        if object_ids:
            qs = qs.filter(object_id__in=object_ids)

        return qs, use_distinct


admin.site.register(LogEntry, LogEntryAdmin)
